<?xml version="1.0" encoding="UTF-8" ?>
<xsl:stylesheet version="1.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:saxon="http://icl.com/saxon"
                extension-element-prefixes="saxon"
                >

<!-- map2htmltoc.xsl   main stylesheet
     Convert DITA map or maplist to a simple HTML table-of-contents view.
     Input = one DITA map file, or a maplist pointing to multiple maps
     Output = one HTML file for viewing/checking by the author.

     Options:
        OUTEXT  = XHTML output extension (default is 'html')
        WORKDIR = The working directory that contains the document being transformed.
                   Needed as a directory prefix for the @href "document()" function calls.
                   Default is './'
-->

<!-- Include error message template -->
<xsl:include href="output-message.xsl"/>

<!-- xsl:output method="saxon:xhtml"/
            encoding="UTF-8"
            doctype-system="http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd"
            doctype-public="-//W3C//DTD XHTML 1.0 Transitional//EN"
-->
<!-- Output XHTML with XML syntax, use UFT-8 encoding="UTF-8", transitional XHTML.
     Prevent indenting to conserve space on output. -->
<!-- <xsl:output method="xml" -->
<xsl:output method="html"
           indent="no"
/>

<!-- Set the prefix for error message numbers -->
<xsl:variable name="msgprefix">IDXS</xsl:variable>

<!-- *************************** Command line parameters *********************** -->
<xsl:param name="OUTEXT" select="'html'"/><!-- "htm" and "html" are valid values -->
<xsl:param name="WORKDIR" select="'./'"/>

<!-- Define a newline character -->
<xsl:variable name="newline"><xsl:text>
</xsl:text></xsl:variable>


<!-- *********************************************************************************
     Setup the HTML wrapper for the table of contents
     ********************************************************************************* -->
<xsl:template match="/">
  <html><xsl:value-of select="$newline"/>
  <head><xsl:value-of select="$newline"/>
   <xsl:if test="/*[contains(@class,' map/map ')]/@title">
    <title><xsl:value-of select="/*[contains(@class,' map/map ')]/@title"></xsl:value-of></title><xsl:value-of select="$newline"/>
   </xsl:if>
  </head><xsl:value-of select="$newline"/>

   <body><xsl:value-of select="$newline"/>
    <xsl:apply-templates/>
   </body><xsl:value-of select="$newline"/>
  </html>
</xsl:template>

<!-- *********************************************************************************
     Process each of the maps in the navigation section.
     ********************************************************************************* -->
<xsl:template match="/maplist">
  <xsl:apply-templates select="nav/map"/>
</xsl:template>

<!-- *********************************************************************************
     If processing only a single map, setup the HTML wrapper and output the contents.
     Otherwise, just process the contents.
     ********************************************************************************* -->
<xsl:template match="/*[contains(@class, ' map/map ')]">
  <xsl:param name="pathFromMaplist"/>
  <xsl:if test="*[contains(@class, ' map/topicref ')][not(@toc='no')]">
    <UL><xsl:value-of select="$newline"/>

      <xsl:apply-templates select="*[contains(@class, ' map/topicref ')][not(@toc='no')]">
        <xsl:with-param name="pathFromMaplist" select="$pathFromMaplist"/>
      </xsl:apply-templates>
    </UL><xsl:value-of select="$newline"/>
  </xsl:if>
</xsl:template>

<!-- *********************************************************************************
     For each map in the navigation secton of the maplist, process the contents.
     ********************************************************************************* -->
<xsl:template match="nav/map">
  <xsl:variable name="MapWithPath"><xsl:value-of select="$WORKDIR"/><xsl:value-of select="@file"/></xsl:variable>
  <xsl:apply-templates select="document($MapWithPath)/*[contains(@class, ' map/map ')]">
    <xsl:with-param name="pathFromMaplist"><xsl:call-template name="getRelativePath"/></xsl:with-param>
  </xsl:apply-templates>
</xsl:template>

<!-- *********************************************************************************
     Output each topic as an <LI> with an A-link. Each item takes 2 values:
     - A title. If a navtitle is specified on <topicref>, use that.
       Otherwise try to open the file and retrieve the title. First look for a navigation title in the topic,
       followed by the main topic title. Last, try to use <linktext> specified in the map.
       Failing that, use *** and issue a message.
     - An HREF is optional. If none is specified, this will generate a wrapper.
       Include the HREF if specified.
     - Ignore if TOC=no.

     If this topicref has any child topicref's that will be part of the navigation,
     output a <UL> around them and process the contents.
     ********************************************************************************* -->
<xsl:template match="*[contains(@class, ' map/topicref ')][not(@toc='no')]">
  <xsl:param name="pathFromMaplist"/>
  <LI>
      <xsl:choose>
        <!-- If there is a reference to a DITA or HTML file, and it is not external: -->
        <xsl:when test="@href">
          <!-- to use an extension other than '.dita' just uncomment the variable and replace
               '.dita' below with $DITAEXT -->
          <!-- <xsl:variable name="DITAEXT">.dita</xsl:variable> -->
          <xsl:element name="a">
            <xsl:attribute name="href">
             <xsl:choose>        <!-- What if targeting a nested topic? Need to keep the ID? -->
              <xsl:when test="contains(@copy-to,'.dita')">
                <xsl:if test="not(@scope='external')"><xsl:value-of select="$pathFromMaplist"/></xsl:if>
                <xsl:value-of select="substring-before(@copy-to,'.dita')"/>.<xsl:value-of select="$OUTEXT"/>
              </xsl:when>
              <xsl:when test="contains(@href,'.dita')">
                <xsl:if test="not(@scope='external')"><xsl:value-of select="$pathFromMaplist"/></xsl:if>
                <xsl:value-of select="substring-before(@href,'.dita')"/>.<xsl:value-of select="$OUTEXT"/>
              </xsl:when>
              <xsl:otherwise>  <!-- If non-DITA, keep the href as-is -->
                <xsl:if test="not(@scope='external')"><xsl:value-of select="$pathFromMaplist"/></xsl:if>
                <xsl:value-of select="@href"/>
              </xsl:otherwise>
             </xsl:choose>
            </xsl:attribute>
           <xsl:call-template name="navtitle"/>
          </xsl:element>
        </xsl:when>

        <xsl:otherwise>
         <xsl:call-template name="navtitle"/>
        </xsl:otherwise>
      </xsl:choose>

       <!-- If there are any children that should be in the TOC, process them -->
       <xsl:if test="*[contains(@class, ' map/topicref ')][not(contains(@toc,'no'))]">
         <xsl:value-of select="$newline"/><UL><xsl:value-of select="$newline"/>
           <xsl:apply-templates select="*[contains(@class, ' map/topicref ')][not(contains(@toc,'no'))]">
             <xsl:with-param name="pathFromMaplist" select="$pathFromMaplist"/>
           </xsl:apply-templates>
         </UL><xsl:value-of select="$newline"/>
       </xsl:if>
  </LI><xsl:value-of select="$newline"/>
</xsl:template>

<xsl:template name="navtitle">
  <xsl:choose>

    <!-- If navtitle is specified, use it (!?but it should only be used when locktitle=yes is specifed?!) -->
    <xsl:when test="@navtitle"><xsl:value-of select="@navtitle"/></xsl:when>

    <!-- If this references a DITA file (has @href, not "local" or "external"),
         try to open the file and get the title -->
    <xsl:when test="@href and
                    not ((ancestor-or-self::*/@scope)[last()]='external') and
                    not ((ancestor-or-self::*/@scope)[last()]='peer') and
                    not ((ancestor-or-self::*/@type)[last()]='external') and
                    not ((ancestor-or-self::*/@type)[last()]='local')">
      <!-- Need to worry about targeting a nested topic? Not for now. -->
      <xsl:variable name="FileWithPath"><xsl:value-of select="$WORKDIR"/><xsl:choose>
        <xsl:when test="@copy-to"><xsl:value-of select="@copy-to"/></xsl:when>
        <xsl:when test="contains(@href,'#')"><xsl:value-of select="substring-before(@href,'#')"/></xsl:when>
        <xsl:otherwise><xsl:value-of select="@href"/></xsl:otherwise></xsl:choose></xsl:variable>
      <xsl:variable name="TargetFile" select="document($FileWithPath)"/>

      <xsl:choose>
        <xsl:when test="not($TargetFile)">   <!-- DITA file does not exist -->
          <xsl:choose>
            <xsl:when test="*[contains(@class, ' map/topicmeta ')]/*[contains(@class, ' map/linktext ')]">  <!-- attempt to recover by using linktext -->
              <xsl:value-of select="*[contains(@class, ' map/topicmeta ')]/*[contains(@class, ' map/linktext ')]"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:call-template name="output-message">
                <xsl:with-param name="msg">File <xsl:value-of select="@href"/> does not exist.</xsl:with-param>
                <xsl:with-param name="msgnum">004</xsl:with-param>
                <xsl:with-param name="msgsev">W</xsl:with-param>
              </xsl:call-template>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:when>
        <!-- First choice for navtitle: topic/titlealts/navtitle -->
        <xsl:when test="$TargetFile/*[contains(@class,' topic/topic ')]/*[contains(@class,' topic/titlealts ')]/*[contains(@class,' topic/navtitle ')]">
          <xsl:value-of select="$TargetFile/*[contains(@class,' topic/topic ')]/*[contains(@class,' topic/titlealts ')]/*[contains(@class,' topic/navtitle ')]"/>
        </xsl:when>
        <!-- Second choice for navtitle: topic/title -->
        <xsl:when test="$TargetFile/*[contains(@class,' topic/topic ')]/*[contains(@class,' topic/title ')]">
          <xsl:value-of select="$TargetFile/*[contains(@class,' topic/topic ')]/*[contains(@class,' topic/title ')]"/>
        </xsl:when>
        <!-- This might be a combo article; modify the same queries: dita/topic/titlealts/navtitle -->
        <xsl:when test="$TargetFile/dita/*[contains(@class,' topic/topic ')]/*[contains(@class,' topic/titlealts ')]/*[contains(@class,' topic/navtitle ')]">
          <xsl:value-of select="$TargetFile/dita/*[contains(@class,' topic/topic ')]/*[contains(@class,' topic/titlealts ')]/*[contains(@class,' topic/navtitle ')]"/>
        </xsl:when>
        <!-- Second choice: dita/topic/title -->
        <xsl:when test="$TargetFile/dita/*[contains(@class,' topic/topic ')]/*[contains(@class,' topic/title ')]">
          <xsl:value-of select="$TargetFile/dita/*[contains(@class,' topic/topic ')]/*[contains(@class,' topic/title ')]"/>
        </xsl:when>
        <!-- Last choice: use the linktext specified within the topicref -->
        <xsl:when test="*[contains(@class, ' map/topicmeta ')]/*[contains(@class, ' map/linktext ')]">
          <xsl:value-of select="*[contains(@class, ' map/topicmeta ')]/*[contains(@class, ' map/linktext ')]"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="output-message">
            <xsl:with-param name="msg">Could not retrieve a title from <xsl:value-of select="$TargetFile"/>. Using '***'.</xsl:with-param>
            <xsl:with-param name="msgnum">005</xsl:with-param>
            <xsl:with-param name="msgsev">W</xsl:with-param>
          </xsl:call-template>
          <xsl:text>***</xsl:text>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>

    <!-- If there is no title and none can be retrieved, check for <linktext> -->
    <xsl:when test="*[contains(@class, ' map/topicmeta ')]/*[contains(@class, ' map/linktext ')]">
      <xsl:value-of select="*[contains(@class, ' map/topicmeta ')]/*[contains(@class, ' map/linktext ')]"/>
    </xsl:when>

    <!-- No local title, and not targeting a DITA file. Could be just a container setting
         metadata, or a file reference with no title. Issue message for the second case. -->
    <xsl:otherwise>
      <xsl:if test="@href">
          <xsl:call-template name="output-message">
            <xsl:with-param name="msg">Could not retrieve a title from <xsl:value-of select="@href"/>. Using the HREF value.</xsl:with-param>
            <xsl:with-param name="msgnum">005</xsl:with-param>
            <xsl:with-param name="msgsev">W</xsl:with-param>
          </xsl:call-template>
          <xsl:value-of select="@href"/>
      </xsl:if>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- These are here just to prevent accidental fallthrough -->
<xsl:template match="*[contains(@class, ' map/navref ')]"/>
<xsl:template match="*[contains(@class, ' map/topicref ')][@toc='no']"/>
<xsl:template match="*[contains(@class, ' map/anchor ')]"/>
<xsl:template match="*[contains(@class, ' map/reltable ')]"/>
<xsl:template match="*[contains(@class, ' map/topicmeta ')]"/>
<xsl:template match="maplist/link"/>

<xsl:template match="*">
  <xsl:apply-templates/>
</xsl:template>

<!-- Convert the input value to lowercase & return it -->
<xsl:template name="convert-to-lower">
 <xsl:param name="inputval"/>
 <xsl:value-of select="translate($inputval,
                                  '-abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ',
                                  '-abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyz')"/>
</xsl:template>

<!-- Template to get the relative path from a maplist to a map -->
<xsl:template name="getRelativePath">
  <xsl:param name="remainingPath" select="@file"/>
  <xsl:choose>
    <xsl:when test="contains($remainingPath,'/')">
      <xsl:value-of select="substring-before($remainingPath,'/')"/><xsl:text>/</xsl:text>
      <xsl:call-template name="getRelativePath">
        <xsl:with-param name="remainingPath" select="substring-after($remainingPath,'/')"/>
      </xsl:call-template>
    </xsl:when>
    <xsl:when test="contains($remainingPath,'\')">
      <xsl:value-of select="substring-before($remainingPath,'\')"/><xsl:text>/</xsl:text>
      <xsl:call-template name="getRelativePath">
        <xsl:with-param name="remainingPath" select="substring-after($remainingPath,'\')"/>
      </xsl:call-template>
    </xsl:when>
  </xsl:choose>
</xsl:template>

</xsl:stylesheet>
