<?xml version="1.0"?>

<!-- *******************************************************************************************

Copyright IBM Corporation, October 2002
Originally developed for IDWB release 3.8 (Spring 2003).

XMLized IBMIDDoc to DITA conversion routine, for use with the Omnimark id2xid script.
INPUT:  One IBMIDDoc file which has been converted to XML using id2xid
OUTPUT: One or more DITA topics
        One ditamap file listing all generated topics, saving the IDDoc hierarchy


                 **************************************************

General notes:
  * Elements are grouped by type. For example, all list elements and their children are in 
    section [4].
  * The parent-element parameter is frequently used. When used, it contains the currently
    open DITA parent, NOT the Ibmiddoc parent (which is easily accessible through parent::*).
  * Many elements cannot always be mapped directly to DITA, and remain in context. These
    elements are placed in <required-cleanup>, and a message is genererated.
  * Many elements need extra wrappers to remain in context, such as <p> or <section>. Because
    of this, many elements have an associated named template of output-standard-ELEMENTNAME.
    The proper wrapper(s) is/are created for each location, and this template is called from within
    the wrapper(s).
  * The mode ensure-proper-placement is frequently used with indices and other elements. If sub-
    elements must be moved, they will be processed with this mode where needed; the same elements
    should be ignored for fallthrough processing.
     
                 **************************************************  

Table of contents:
[0] Root Rule
[1] General Purpose Functions
  [1a] Set the DOCTYPE for the post-processor
  [1b] Functions to add attributes used by many elements
  [1c] Get the ID, file or infotype of the current 'topic'
  [1d] Output a warning message when required-cleanup is created, or an element unsupported
  [1e] Generate a map, starting with the current node
  [1f] Pull index entries into a definition
  [1z] Other general purpose functions
[2] Prolog, metadata templates for map and for topics
  [2a] Dintro/Abstract mapping to shortdesc
  [2b] Authors                      [2c] Copyright
  [2d] Critdates                    [2e] Publisher (map only)
  [2f] Permissions (topic only)     [2g] Audience (topic only)
  [2h] Category  (topic only)       [2i] Keywords (topic only)
  [2j] Prodinfo                     [2k] Othermeta
  [2z] Empty templates to prevent prolog fallthrough
[3] Division level elements
  [3a] Output a topic's title       [3b] Output the titlealts
  [3c] Output the shortdesc         [3d] Output the prolog
  [3e] Output the body              [3e.1] Output a task body
  [3f] Output the contents of a topic
  [3g] General function to output a divlike element
[4] Lists
[5] Tables
[6] Figures
[7] Notes
[8] Titles
[9] Non-wrapping elements (xmp, screen, etc)
[10] LERS
[11] Syntax
[12] Link/XREF/LQ
[13] PROC
[14] Images, Imagedescs
[15] ASMList, PartASM
[16] Message lists
[17] Indexing
  [17a] Common functions dealing with indices
  [17b] Process inline i1, i2 and i3 elements
  [17c] Process referenced i1, i2 and i3 elements
  [17d] Process iref
[18] ModInfo
[19] CITE
[20] BibEntry, LibEntry
[21] OBJLIB
[22] General block-like elements
[23] Generatl phrase-like elements
[24] Smalldoc
[25] Recognized Processing Instructions


Bugs that still need to be worked out:
   * NNNdef elements are only referenced in /ibmiddoc/prolog, not in dprolog or specdprolog.
     Probably can use ancestor::*/*[contains(name(),'prolog')] instead of /ibmiddoc/prolog
     Also - need to put the xpath in paren's, and add [last()] to the end: gets the closest ancestor.
   * output-message marks places that need messages, or (in a few cases) need code updates

                 **************************************************                             -->

<!DOCTYPE xsl:stylesheet [

  <!ENTITY gt            "&gt;">
  <!ENTITY lt            "&lt;">
  <!ENTITY rbl           " ">
  <!ENTITY nbsp          "&#xA0;">    <!-- &#160; -->
  <!ENTITY quot          "&#34;">
  <!ENTITY copyr         "&#169;">
]>

<xsl:stylesheet version="1.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:saxon="http://icl.com/saxon"
                xmlns:xt="http://www.jclark.com/xt"
                extension-element-prefixes="saxon xt">

<!-- Include error message template -->
<xsl:include href="output-message.xsl"/>

<xsl:output method="xml"
            encoding="utf-8"
/>

<!-- Set the prefix for error message numbers -->
<xsl:variable name="msgprefix">IDXS</xsl:variable>

<!-- *************************** Command line parameters *********************** -->
<xsl:param name="WORKDIR" select="'./'"/>
<xsl:param name="FILENAME" select="'input.xid'"/>
<xsl:param name="MSGLIST" select="'newfile'"/>  <!-- how to format msglist: newfile or dl -->
<xsl:param name="YEAR">2002</xsl:param>

<!-- ***************************** Any other variables ************************* -->
<xsl:variable name="BASEFILENAME">
  <xsl:value-of select="substring-before($FILENAME,'.xid')"/>
</xsl:variable>

<!-- The language will be added to every topic; keep it here, so we don't re-evaluate it -->
<xsl:variable name="bookLanguage">
  <xsl:variable name="booklang"><xsl:value-of select="/ibmiddoc/@language"/></xsl:variable>
  <xsl:choose>
    <xsl:when test="not(/ibmiddoc/@language)">en-us</xsl:when>
    <xsl:when test="$booklang='english' or $booklang='en_us' or $booklang='usenglish'">en-us</xsl:when>
    <xsl:when test="$booklang='arabic' or $booklang='ar_eg'">ar-eg</xsl:when>
    <xsl:when test="$booklang='bdutch' or $booklang='nl_be'">nl-be</xsl:when>
    <xsl:when test="$booklang='bfrench' or $booklang='fr_be'">fr-be</xsl:when>
    <xsl:when test="$booklang='bportuguese' or $booklang='pt_br'">pt-br</xsl:when>
    <xsl:when test="$booklang='bulgarian' or $booklang='bg_bg'">bg-bg</xsl:when>
    <xsl:when test="$booklang='catalan' or $booklang='ca_es'">ca-es</xsl:when>
    <xsl:when test="$booklang='cenglish' or $booklang='en_ca'">en-ca</xsl:when>
    <xsl:when test="$booklang='cfrench' or $booklang='fr_ca'">fr-ca</xsl:when>
    <xsl:when test="$booklang='croatian' or $booklang='hr_hr'">hr-hr</xsl:when>
    <xsl:when test="$booklang='czech' or $booklang='cs_cz'">cs-cz</xsl:when>
    <xsl:when test="$booklang='danish' or $booklang='da_dk'">da-dk</xsl:when>
    <xsl:when test="$booklang='dutch' or $booklang='nl_nl'">nl-nl</xsl:when>
    <xsl:when test="$booklang='estonian' or $booklang='et_ee'">et-ee</xsl:when>
    <xsl:when test="$booklang='finnish' or $booklang='fi_fi'">fi-fi</xsl:when>
    <xsl:when test="$booklang='french' or $booklang='fr_fr'">fr-fr</xsl:when>
    <xsl:when test="$booklang='german' or $booklang='de_de'">de-de</xsl:when>
    <xsl:when test="$booklang='greek' or $booklang='el_gr'">el-gr</xsl:when>
    <xsl:when test="$booklang='hebrew' or $booklang='he_il'">he-il</xsl:when>
    <xsl:when test="$booklang='hungarian' or $booklang='hu_hu'">hu-hu</xsl:when>
    <xsl:when test="$booklang='icelandic' or $booklang='is_is'">is-is</xsl:when>
    <xsl:when test="$booklang='italian' or $booklang='it_it'">it-it</xsl:when>
    <xsl:when test="$booklang='japanese' or $booklang='ja_jp'">ja-jp</xsl:when>
    <xsl:when test="$booklang='korean' or $booklang='ko_kr'">ko-kr</xsl:when>
    <xsl:when test="$booklang='latvian' or $booklang='lv_lv'">lv-lv</xsl:when>
    <xsl:when test="$booklang='lithuanian' or $booklang='lt_lt'">lt-lt</xsl:when>
    <xsl:when test="$booklang='macedonian' or $booklang='mk_mk'">mk-mk</xsl:when>
    <xsl:when test="$booklang='norwegian' or $booklang='no_no'">no-no</xsl:when>
    <xsl:when test="$booklang='polish' or $booklang='pl_pl'">pl-pl</xsl:when>
    <xsl:when test="$booklang='portuguese' or $booklang='pt_pt'">pt-pt</xsl:when>
    <xsl:when test="$booklang='romainian' or $booklang='ro_ro'">ro-ro</xsl:when>
    <xsl:when test="$booklang='russian' or $booklang='ru_ru'">ru-ru</xsl:when>
    <xsl:when test="$booklang='schinese' or $booklang='zh_cn'">zh-cn</xsl:when>
    <xsl:when test="$booklang='serbian' or $booklang='sr_sp'">sr-sp</xsl:when>
    <xsl:when test="$booklang='sfrench' or $booklang='fr_ch'">fr-ch</xsl:when>
    <xsl:when test="$booklang='sgerman' or $booklang='de_ch'">de-ch</xsl:when>
    <xsl:when test="$booklang='sitalian' or $booklang='it_ch'">it-ch</xsl:when>
    <xsl:when test="$booklang='slovak' or $booklang='sk_sk'">sk-sk</xsl:when>
    <xsl:when test="$booklang='slovenian' or $booklang='sl_si'">sl-si</xsl:when>
    <xsl:when test="$booklang='spanish' or $booklang='es_es'">es-es</xsl:when>
    <xsl:when test="$booklang='swedish' or $booklang='sv_se'">sv-se</xsl:when>
    <xsl:when test="$booklang='tchinese' or $booklang='zh_tw'">zh-tw</xsl:when>
    <xsl:when test="$booklang='thai' or $booklang='th_th'">th-th</xsl:when>
    <xsl:when test="$booklang='turkish' or $booklang='tr_tr'">tr-tr</xsl:when>
    <xsl:when test="$booklang='ukenglish' or $booklang='en_gb'">en-gb</xsl:when>
    <xsl:otherwise>en-us</xsl:otherwise>
  </xsl:choose>
</xsl:variable>

<!-- Define a newline character -->
<xsl:variable name="newline"><xsl:text>
</xsl:text></xsl:variable>


<!-- **************************** [0] ROOT RULE **************************************
     ********************************************************************************* -->
<xsl:template match="/">
  <xsl:apply-templates/>                   <!-- Generate topics -->
  <xsl:call-template name="generate-map"/> <!-- Generate the map -->
</xsl:template>

<!-- Document title - go to map title? -->
<xsl:template match="prolog/ibmbibentry/doctitle/titleblk/title">
</xsl:template>

<!-- ******************** [1] GENERAL PURPOSE FUNCTIONS ****************** -->

<!-- [1a] Set the DOCTYPE using a PI that the next stage can convert. The newlines ensure
          this PI will be on its own line, to make REXX processing easier. -->
<xsl:template name="use-concept-dtd">
  <xsl:value-of select="$newline"/>
  <xsl:processing-instruction name="dtd-update">!DOCTYPE concept PUBLIC "-//IBM//DTD DITA Concept//EN" "..\dtd\concept.dtd"</xsl:processing-instruction>
  <xsl:value-of select="$newline"/>
</xsl:template>
<xsl:template name="use-task-dtd">
  <xsl:value-of select="$newline"/>
  <xsl:processing-instruction name="dtd-update">!DOCTYPE task PUBLIC "-//IBM//DTD DITA Task//EN" "..\dtd\task.dtd"</xsl:processing-instruction>
  <xsl:value-of select="$newline"/>
</xsl:template>
<xsl:template name="use-reference-dtd">
  <xsl:value-of select="$newline"/>
  <xsl:processing-instruction name="dtd-update">!DOCTYPE reference PUBLIC "-//IBM//DTD DITA Reference//EN" "..\dtd\reference.dtd"</xsl:processing-instruction>
  <xsl:value-of select="$newline"/>
</xsl:template>
<xsl:template name="use-topic-dtd">
  <xsl:value-of select="$newline"/>
  <xsl:processing-instruction name="dtd-update">!DOCTYPE topic PUBLIC "-//IBM//DTD DITA Topic//EN" "..\dtd\topic.dtd"</xsl:processing-instruction>
  <xsl:value-of select="$newline"/>
</xsl:template>
<xsl:template name="use-dita-dtd">
  <xsl:value-of select="$newline"/>
  <xsl:processing-instruction name="dtd-update">!DOCTYPE dita PUBLIC "-//IBM//DTD DITA Composite//EN" "..\dtd\ditabase.dtd"</xsl:processing-instruction>
  <xsl:value-of select="$newline"/>
</xsl:template>
<xsl:template name="use-map-dtd">
  <xsl:value-of select="$newline"/>
  <xsl:processing-instruction name="dtd-update">!DOCTYPE map PUBLIC "-//IBM//DTD DITA Map//EN" "..\dtd\map.dtd"</xsl:processing-instruction>
  <xsl:value-of select="$newline"/>
</xsl:template>

<!-- [1b] Add attributes used by many elements -->
<!-- Add the default attributes that any element can take: 
                     id, outputclass, translate, conref, otherprops 
     E003796: Use apply-templates for each attribute, to make overriding easier. -->
<xsl:template name="add-default-attributes">
  <xsl:apply-templates select="@id"/>
  <xsl:apply-templates select="@class"/>
  <xsl:apply-templates select="@translate"/>
  <xsl:apply-templates select="@otherprops"/>
  <xsl:apply-templates select="@conloc"/>
  <xsl:apply-templates select="@rev"/>
</xsl:template>

<!-- Templates to catch each of the attributes checked in [1b] -->
<xsl:template match="@id">
  <xsl:attribute name="id"><xsl:value-of select="."/></xsl:attribute>
</xsl:template>
<xsl:template match="@class">
  <xsl:if test="key('classdef',.)/@outputclass">
    <xsl:attribute name="outputclass">
      <xsl:value-of select="key('classdef',.)/@outputclass"/>
    </xsl:attribute>
  </xsl:if>
</xsl:template>
<xsl:template match="@translate">
  <xsl:attribute name="translate">
    <xsl:choose>
      <xsl:when test=".='translate'">yes</xsl:when>
      <xsl:otherwise>no</xsl:otherwise>
    </xsl:choose>
  </xsl:attribute>
</xsl:template>
<xsl:template match="@otherprops">
  <xsl:attribute name="otherprops"><xsl:value-of select="."/></xsl:attribute>
</xsl:template>
<xsl:template match="@conloc">
  <!-- Convert conloc to conref, using the xref format for the target ID -->
  <xsl:variable name="thisfile"><xsl:call-template name="get-current-chunk-name"/></xsl:variable>
  <xsl:attribute name="conref">
    <!-- Only process the first one, in case there are multiple instances of the ID -->
    <xsl:apply-templates select="key('xreflist',.)[1]" mode="get-xref-target">
      <xsl:with-param name="xreffile"><xsl:value-of select="$thisfile"/></xsl:with-param>
    </xsl:apply-templates>
  </xsl:attribute>
</xsl:template>
<xsl:template match="@rev"> <!-- Added for E003796 -->
  <!-- Not valid on some elements, like <body>, but they should not add defaults -->
  <xsl:attribute name="rev"><xsl:value-of select="."/></xsl:attribute>
</xsl:template>

<!-- Add the frame attribute to elements that use %display-atts; 
     Value can be specified locally, or inherited from a def element (currently only figdef) -->
<xsl:template name="add-frame-attribute">
  <xsl:variable name="frameval">
    <xsl:choose>
      <xsl:when test="@frame">
        <xsl:value-of select="@frame"/>
      </xsl:when>
      <xsl:when test="self::fig and @def and /ibmiddoc/prolog/propdefs/figdef[@defname=current()/@def]/@frame">
        <xsl:value-of select="/ibmiddoc/prolog/propdefs/figdef[@defname=current()/@def]/@frame"/>
      </xsl:when>
      <xsl:when test="self::fig and /ibmiddoc/prolog/propdefs/figdef[not(@defname)]/@frame">
        <xsl:value-of select="/ibmiddoc/prolog/propdefs/figdef[not(@defname)]/@frame"/>
      </xsl:when>
    </xsl:choose>
  </xsl:variable>
  <xsl:if test="string-length($frameval) > 0">
    <xsl:attribute name="frame">
      <xsl:choose>
        <xsl:when test="contains($frameval,'box')">all</xsl:when>
        <xsl:when test="contains($frameval,'rules')">topbot</xsl:when>
        <xsl:when test="contains($frameval,'none')">none</xsl:when>
        <xsl:otherwise>none</xsl:otherwise>  <!-- user has an unknown value; should not happen -->
      </xsl:choose>
    </xsl:attribute>
  </xsl:if>
</xsl:template>

<!-- Add the scale attribute to elements that use %display-atts; -->
<xsl:template name="add-scale-attribute">
  <xsl:variable name="scaleval">
    <xsl:choose>
      <xsl:when test="@scalepct">
        <xsl:value-of select="@scalepct"/>
      </xsl:when>
      <xsl:when test="self::fig and @def and /ibmiddoc/prolog/propdefs/figdef[@defname=current()/@def]/@scalepct">
        <xsl:value-of select="/ibmiddoc/prolog/propdefs/figdef[@defname=current()/@def]/@scalepct"/>
      </xsl:when>
      <xsl:when test="self::fig and /ibmiddoc/prolog/propdefs/figdef[not(@defname)]/@scalepct">
        <xsl:value-of select="/ibmiddoc/prolog/propdefs/figdef[not(@defname)]/@scalepct"/>
      </xsl:when>
      <xsl:when test="self::syntax and @def and /ibmiddoc/prolog/propdefs/syntaxdef[@defname=current()/@def]/@scalepct">
        <xsl:value-of select="/ibmiddoc/prolog/propdefs/syntaxdef[@defname=current()/@def]/@scalepct"/>
      </xsl:when>
      <xsl:when test="self::syntax and /ibmiddoc/prolog/propdefs/syntaxdef[not(@defname)]/@scalepct">
        <xsl:value-of select="/ibmiddoc/prolog/propdefs/syntaxdef[not(@defname)]/@scalepct"/>
      </xsl:when>
    </xsl:choose>
  </xsl:variable>
  <!-- IDDoc scale value can take any number, but DITA has limited values. Round the IDDoc number to
       the closest available DITA value. -->
  <xsl:variable name="actualscaleval">
    <xsl:choose>
      <xsl:when test="not(@scale)"/>
      <xsl:when test="number($scaleval)=50 or number($scaleval)=60 or number($scaleval)=70 or
                      number($scaleval)=80 or number($scaleval)=90 or number($scaleval)=100 or
                      number($scaleval)=110 or number($scaleval)=120 or number($scaleval)=140 or
                      number($scaleval)=160 or number($scaleval)=180 or number($scaleval)=200">
        <xsl:value-of select="$scaleval"/>
      </xsl:when>
      <xsl:otherwise>
        <!-- output-message: invalid scale value -->
        <xsl:choose>
          <xsl:when test="number($scaleval) &lt; 55">50</xsl:when>
          <xsl:when test="number($scaleval) &lt; 65">60</xsl:when>
          <xsl:when test="number($scaleval) &lt; 75">70</xsl:when>
          <xsl:when test="number($scaleval) &lt; 85">80</xsl:when>
          <xsl:when test="number($scaleval) &lt; 95">90</xsl:when>
          <xsl:when test="number($scaleval) &lt; 105">100</xsl:when>
          <xsl:when test="number($scaleval) &lt; 115">110</xsl:when>
          <xsl:when test="number($scaleval) &lt; 130">120</xsl:when>
          <xsl:when test="number($scaleval) &lt; 150">140</xsl:when>
          <xsl:when test="number($scaleval) &lt; 170">160</xsl:when>
          <xsl:when test="number($scaleval) &lt; 190">180</xsl:when>
          <xsl:otherwise>200</xsl:otherwise>
        </xsl:choose>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:if test="string-length($actualscaleval) > 0">
    <xsl:attribute name="scale"><xsl:value-of select="$actualscaleval"/></xsl:attribute>
  </xsl:if>
</xsl:template>

<!-- Convert the optreq attribute on syntax elements to DITA's importance attribute -->
<xsl:template match="@optreq">
  <xsl:attribute name="importance">
    <xsl:choose>
      <xsl:when test=".='def'">default</xsl:when>
      <xsl:when test=".='req'">required</xsl:when>
      <xsl:otherwise>optional</xsl:otherwise>
    </xsl:choose>
  </xsl:attribute>
</xsl:template>
<!--<xsl:template name="add-importance-attribute">
  <xsl:if test="@optreq">
    <xsl:attribute name="importance">
      <xsl:choose>
        <xsl:when test="@optreq='def'">default</xsl:when>
        <xsl:when test="@optreq='req'">required</xsl:when>
        <xsl:otherwise>optional</xsl:otherwise>
      </xsl:choose>
    </xsl:attribute>
  </xsl:if>
</xsl:template>-->

<!-- The pgwide attribute is evaluated differently on every element. 
     But, we can have a common process for retrieving it. This function finds the current
     pgwide value, and returns it; it does not attempt to create the attribute. -->
<xsl:template name="get-pgwide-attribute">
  <xsl:choose>
    <xsl:when test="@pgwide"><xsl:value-of select="@pgwide"/></xsl:when>
    <xsl:when test="self::fig and @def and /ibmiddoc/prolog/propdefs/figdef[@defname=current()/@def]/@pgwide">
      <xsl:value-of select="/ibmiddoc/prolog/propdefs/figdef[@defname=current()/@def]/@pgwide"/>
    </xsl:when>
    <xsl:when test="self::fig and ancestor::*/*[contains(name(),'prolog')]/propdefs/figdef[not(@defname)]/@pgwide">
      <xsl:value-of select="(ancestor::*/*[contains(name(),'prolog')]/propdefs/figdef[not(@defname)]/@pgwide)[last()]"/>
    </xsl:when>
    <xsl:when test="self::fig and /ibmiddoc/prolog/propdefs/figdef[not(@defname)]/@pgwide">
      <xsl:value-of select="/ibmiddoc/prolog/propdefs/figdef[not(@defname)]/@pgwide"/>
    </xsl:when>
    <xsl:when test="self::xmp and @def and /ibmiddoc/prolog/propdefs/xmpdef[@defname=current()/@def]/@pgwide">
      <xsl:value-of select="/ibmiddoc/prolog/propdefs/xmpdef[@defname=current()/@def]/@pgwide"/>
    </xsl:when>
    <xsl:when test="self::xmp and /ibmiddoc/prolog/propdefs/xmpdef[not(@defname)]/@pgwide">
      <xsl:value-of select="/ibmiddoc/prolog/propdefs/xmpdef[not(@defname)]/@pgwide"/>
    </xsl:when>
    <xsl:when test="self::screen and @def and /ibmiddoc/prolog/propdefs/screendef[@defname=current()/@def]/@pgwide">
      <xsl:value-of select="/ibmiddoc/prolog/propdefs/screendef[@defname=current()/@def]/@pgwide"/>
    </xsl:when>
    <xsl:when test="self::screen and /ibmiddoc/prolog/propdefs/screendef[not(@defname)]/@pgwide">
      <xsl:value-of select="/ibmiddoc/prolog/propdefs/screendef[not(@defname)]/@pgwide"/>
    </xsl:when>
    <xsl:when test="self::syntax and @def and /ibmiddoc/prolog/propdefs/syntaxdef[@defname=current()/@def]/@pgwide">
      <xsl:value-of select="/ibmiddoc/prolog/propdefs/syntaxdef[@defname=current()/@def]/@pgwide"/>
    </xsl:when>
    <xsl:when test="self::syntax and /ibmiddoc/prolog/propdefs/syntaxdef[not(@defname)]/@pgwide">
      <xsl:value-of select="/ibmiddoc/prolog/propdefs/syntaxdef[not(@defname)]/@pgwide"/>
    </xsl:when>
  </xsl:choose>
</xsl:template>

<!-- Get the language of the IDD book. It should be added to each topic. -->
<xsl:template name="add-language-attribute">
  <xsl:attribute name="xml:lang"><xsl:value-of select="$bookLanguage"/></xsl:attribute>
</xsl:template>

<!-- [1c] Get the ID or file of the current 'topic' -->
<!-- divlike elements are: d | abbrev | bibliog | glossary | legend | preface | safety | soa | 
                     frontm/abstract | ednotices | notices | le | mod | proc | objlib | part | 
                     modinfo | msg | partasm | ... -->
<!-- Post beta3: Replaced the name()= tests with @chunk-filename: anything with that attribute
                 will map to a topic, so use its ID. Also, can remove special frontm/abstract handling. -->
<xsl:template name="get-current-topic-id">
  <xsl:choose>
    <xsl:when test="ancestor-or-self::*[@chunk-filename]">
      <!-- Select the most recent divlike ancestor. If it has an ID, use that, otherwise the genid -->
      <xsl:for-each select="ancestor-or-self::*[@chunk-filename][1]">
        <xsl:choose>
          <xsl:when test="@id"><xsl:value-of select="@id"/></xsl:when>
          <xsl:otherwise><xsl:value-of select="generate-id(.)"/></xsl:otherwise>
        </xsl:choose>
      </xsl:for-each>
    </xsl:when>
    <!-- If this is a cross-book link, there will be a warning; use an ID that indicates the problem -->
    <xsl:when test="self::nameloc">unresolved-cross-book-link</xsl:when>
    <!-- Cannot find a div-like parent; shouldn't happen -->
    <xsl:otherwise>unresolvedid</xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Get the name of the current file; it is stored in @chunk-filename on the closest ancestor -->
<xsl:template name="get-current-chunk-name">
  <xsl:value-of select="ancestor-or-self::*[@chunk-filename][1]/@chunk-filename"/>
</xsl:template>

<!-- Find the infotype when in a division or special division. Added a check for elements with defaults,
     and for elements that always map the same way (like proc), so this can always be used to find
     the current topic type. -->
<xsl:template name="determine-topic-type">
  <xsl:param name="default-type">topic</xsl:param>
  <xsl:choose>
    <!-- These elements always have the same mapping: the type will not vary based on metadata or @class -->
    <xsl:when test="self::msg">reference</xsl:when>
    <xsl:when test="self::proc">task</xsl:when>
    <xsl:when test="self::mod and not(parent::modinfo[@style='table'])">reference</xsl:when>
    <xsl:when test="self::modinfo">reference</xsl:when>
    <!-- For remaining elements, check metadata element first -->
    <xsl:when test="contains(dprolog/metadata/@type,'concept') or 
                    contains(specdprolog/metadata/@type,'concept')">
      <xsl:text>concept</xsl:text>
    </xsl:when>
    <xsl:when test="contains(dprolog/metadata/@type,'task') or 
                    contains(specdprolog/metadata/@type,'task')">
      <xsl:text>task</xsl:text>
    </xsl:when>
    <xsl:when test="contains(dprolog/metadata/@type,'reference') or 
                    contains(specdprolog/metadata/@type,'reference')">
      <xsl:text>reference</xsl:text>
    </xsl:when>
    <xsl:when test="contains(dprolog/metadata/@type,'advisor') or 
                    contains(specdprolog/metadata/@type,'advisor')">
      <xsl:text>topic</xsl:text>
    </xsl:when>
    <xsl:when test="contains(dprolog/metadata/@type,'definition') or 
                    contains(specdprolog/metadata/@type,'definition')">
      <xsl:text>concept</xsl:text>
    </xsl:when>
    <xsl:when test="contains(dprolog/metadata/@type,'example') or 
                    contains(specdprolog/metadata/@type,'example')">
      <xsl:text>topic</xsl:text>
    </xsl:when>
    <xsl:when test="contains(dprolog/metadata/@type,'wizard') or 
                    contains(specdprolog/metadata/@type,'wizard')">
      <xsl:text>topic</xsl:text>
    </xsl:when>
    <!-- Next check class attribute -->
    <xsl:when test="contains(@class,'concept')">concept</xsl:when>
    <xsl:when test="contains(@class,'task')">task</xsl:when>
    <xsl:when test="contains(@class,'procedure')">task</xsl:when>
    <xsl:when test="contains(@class,'reference')">reference</xsl:when>
    <!-- Now check for elements with known defaults defaults -->
    <xsl:when test="self::le">reference</xsl:when>
    <!-- Otherwise, default to passed in value -->
    <xsl:when test="contains($default-type,'concept')">concept</xsl:when>
    <xsl:when test="contains($default-type,'task')">task</xsl:when>
    <xsl:when test="contains($default-type,'reference')">reference</xsl:when>
    <xsl:otherwise>topic</xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- [1d] Output a message any time required-cleanup is created, or an element is unsupported -->
<xsl:template name="output-required-cleanup-warning">
  <xsl:variable name="current-file">
    <xsl:call-template name="get-current-chunk-name"/>
  </xsl:variable>
  <xsl:call-template name="output-message">
    <xsl:with-param name="msg"><xsl:value-of select="$current-file"/> contains 'required-cleanup' from a <xsl:value-of select="name()"/> element that
could not be mapped into a valid DITA element. You should open the 
DITA file and fix the markup.</xsl:with-param>
    <xsl:with-param name="msgnum">013</xsl:with-param>
    <xsl:with-param name="msgsev">W</xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template name="unsupported-element">
  <xsl:call-template name="output-message">
    <xsl:with-param name="msg">The <xsl:value-of select="name()"/> element is not supported by the IBMIDDoc to
DITA migration. The element will be ignored.</xsl:with-param>
    <xsl:with-param name="msgnum">038</xsl:with-param>
    <xsl:with-param name="msgsev">W</xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!-- [1e] Generate a map, starting with the current node. May be able to add a parameter, and
          use this to generate PTOC maps, but it currntly only uses the base file name. -->
<xsl:template name="generate-map">
  <saxon:output href="{$BASEFILENAME}.ditamap">
    <xsl:call-template name="use-map-dtd"/>
    <map>
      <xsl:choose>
        <xsl:when test="string-length(/ibmiddoc/prolog/ibmbibentry/doctitle/titleblk/title)>0">
          <xsl:attribute name="title">
            <xsl:value-of select="/ibmiddoc/prolog/ibmbibentry/doctitle/titleblk/title"/>
          </xsl:attribute>
        </xsl:when>
        <xsl:when test="/ibmiddoc/frontm/toc/titleblk/title">
          <xsl:attribute name="title">
            <xsl:value-of select="/ibmiddoc/frontm/toc/titleblk/title"/>
          </xsl:attribute>
        </xsl:when>
      </xsl:choose>
      <xsl:call-template name="generate-metadata-in-map"/>
      <!-- When a manual TOC is used, we still want the default TOC that lists every topic.
           The create-map mode runs through the entire hierarchy, and creates a topicref
           for every div-like element (everything that chunks). -->
      <xsl:apply-templates mode="create-map">
        <xsl:with-param name="current-depth" select="1"/>
        <xsl:with-param name="maxtoc"><xsl:value-of select="/ibmiddoc/@maxtoc"/></xsl:with-param>
      </xsl:apply-templates>
      <xsl:value-of select="$newline"/>
    </map>
  </saxon:output>
</xsl:template>

<!-- Add metadata to the map.This is called for the default map that lists every topic,
     and for any user-generated map based on CLE. -->
<xsl:template name="generate-metadata-in-map">
  <!-- Always create topicmeta, instead of doing a long complex check -->
  <xsl:value-of select="$newline"/>
  <topicmeta>
    <!-- Call a named template to create each metadata element -->
    <xsl:call-template name="output-authors-metadata-in-map"/>
    <!-- source metadata? -->
    <xsl:call-template name="output-publisher-metadata-in-map"/>
    <xsl:call-template name="output-copyright-metadata-in-map"/>
    <xsl:call-template name="output-critdates-metadata-in-map"/>
    <!-- permissions metadata? -->
    <xsl:call-template name="output-prodinfo-metadata-in-map"/>
    <xsl:call-template name="output-ibmfeatnum-as-othermeta-in-map"/>
    <xsl:call-template name="output-ibmpgmnum-as-othermeta-in-map"/>
    <xsl:call-template name="output-filenum-as-othermeta-in-map"/>
    <xsl:call-template name="output-ibmpartnum-as-othermeta-in-map"/>
    <xsl:call-template name="output-ibmdocnum-as-othermeta-in-map"/>
    <xsl:call-template name="output-origibmdocnum-as-othermeta-in-map"/>
    <xsl:call-template name="output-prtloc-as-othermeta-in-map"/>
    <xsl:call-template name="output-publicid-as-othermeta-in-map"/>
    <xsl:call-template name="output-volid-as-othermeta-in-map"/>
  </topicmeta>
</xsl:template>

<!-- When creating a map, match anything with a chunk-filename and use it to create a topicref.
     If the depth is greater than MAXTOC allows for the division, add @toc=no.
     Ignore the TOC element, which may have chunk-filename to create a manual TOC. -->
<xsl:template match="*[@chunk-filename][not(self::toc)]" mode="create-map">
  <xsl:param name="current-depth" select="1"/>
  <xsl:param name="maxtoc"/>
  <!-- Find the type for this chunk, so @type can be created.
       UPDATE: the type attribute should not be set. It should be picked up dynamically
       when processing the map. -->
  <!--<xsl:variable name="infotype">
    <xsl:call-template name="determine-topic-type">
      <xsl:with-param name="default-type">topic</xsl:with-param>
    </xsl:call-template>
  </xsl:variable>-->
  
  <xsl:choose>
    
    <!-- When @conloc is specified, the referenced division (and children) should be in the map. -->
    <xsl:when test="@conloc">
      <xsl:apply-templates select="key('xreflist',@conloc)[1]" mode="create-map">
        <xsl:with-param name="current-depth">
          <!-- do not increase the depth, because essentially we are re-processing the current topic -->
          <xsl:value-of select="$current-depth"/>
        </xsl:with-param>
        <xsl:with-param name="maxtoc"><xsl:value-of select="$maxtoc"/></xsl:with-param>
      </xsl:apply-templates>
    </xsl:when>
    
    <!-- If this is the object library, BUT every child in the objlibbody will be chunked
         to a unique file, do not include this in the map. The objlib should only be included
         in the map if it will actually be created (if it contains block or phrase level items). -->
    <xsl:when test="self::objlib and not(objlibbody/*[not(@chunk-filename)][not(self::dblk or self::desc)])"/>
    
    <!-- Modinfo and mod should appear in the TOC if they are mapping to chunks,
         but not if they are mapping to a table -->
    <xsl:when test="self::modinfo and @style='table'">
      <xsl:apply-templates mode="create-map">
        <xsl:with-param name="current-depth" select="$current-depth"/>
        <xsl:with-param name="maxtoc"><xsl:value-of select="$maxtoc"/></xsl:with-param>
      </xsl:apply-templates>
    </xsl:when>
    <xsl:when test="parent::modinfo/@style='table'">
      <xsl:apply-templates mode="create-map">
        <xsl:with-param name="current-depth" select="$current-depth"/>
        <xsl:with-param name="maxtoc"><xsl:value-of select="$maxtoc"/></xsl:with-param>
      </xsl:apply-templates>
    </xsl:when>
    
    <xsl:otherwise>
      <xsl:value-of select="$newline"/>
      <topicref>
        <xsl:attribute name="href"><xsl:value-of select="@chunk-filename"/></xsl:attribute>
        <!-- The type attribute should not be getting set; it should be picked up later. -->
        <!--<xsl:attribute name="type"><xsl:value-of select="$infotype"/></xsl:attribute>-->
        <xsl:call-template name="add-navtitle-attribute"/>
        <xsl:if test="@otherprops">
          <xsl:attribute name="otherprops"><xsl:value-of select="@otherprops"/></xsl:attribute>
        </xsl:if>
        <xsl:choose>         <!-- Create the toc attribute -->
          <xsl:when test="@toc='toc'"><xsl:attribute name="toc">yes</xsl:attribute></xsl:when>
          <xsl:when test="@toc='notoc'"><xsl:attribute name="toc">no</xsl:attribute></xsl:when>
          <xsl:when test="string-length($maxtoc)>0">
            <xsl:if test="number($current-depth) > number($maxtoc)">
              <xsl:attribute name="toc">no</xsl:attribute>
            </xsl:if>
          </xsl:when>
        </xsl:choose>
        <xsl:choose>
          <!-- The objlib should only be in one file. Add it to the map, with toc=no and print=no. 
               Do not process children: any children that map to divisions will be included in the
               map when referenced. -->
          <xsl:when test="self::objlib">
            <xsl:attribute name="toc">no</xsl:attribute>
            <xsl:attribute name="print">no</xsl:attribute>
          </xsl:when>
          <!-- Any other element outside of objlib could have nested chunks, so continue processing. -->
          <xsl:otherwise>
            <xsl:if test="self::longdesc">  <!-- Image descriptions should not be in the TOC -->
              <xsl:attribute name="toc">no</xsl:attribute>
              <xsl:attribute name="print">no</xsl:attribute>
            </xsl:if>
            <xsl:apply-templates mode="create-map">
              <xsl:with-param name="current-depth">
                <xsl:choose>
                  <xsl:when test="self::part">0</xsl:when>
                  <xsl:otherwise><xsl:value-of select="$current-depth + 1"/></xsl:otherwise>
                </xsl:choose>
              </xsl:with-param>
              <xsl:with-param name="maxtoc"><xsl:value-of select="$maxtoc"/></xsl:with-param>
            </xsl:apply-templates>
          </xsl:otherwise>
        </xsl:choose>
        <xsl:value-of select="$newline"/>
      </topicref>
    </xsl:otherwise>

  </xsl:choose>
</xsl:template>

<!-- Create a navtitle attribute for the topicref. This is called from div-like elements when
     creating a topicref. First try to grab the stitle, if available. If not, select the title
     based on the current element. Fallback to titleblk or title for unlisted elements (should
     not occur). 
     P016829: Use apply-templates with mode=navtitle, instead of value-of. -->
<xsl:template name="add-navtitle-attribute">
  <xsl:variable name="navtitle-val">
    <xsl:choose>
      <xsl:when test="dprolog/titleblk/stitle">
        <xsl:apply-templates select="dprolog/titleblk/stitle" mode="navtitle"/>
      </xsl:when>
      <xsl:when test="specdprolog/titleblk/stitle">
        <xsl:apply-templates select="specdprolog/titleblk/stitle" mode="navtitle"/>
      </xsl:when>
      <xsl:when test="dprolog/titleblk/title">          <!-- standard division -->
        <xsl:apply-templates select="dprolog/titleblk/title" mode="navtitle"/>
      </xsl:when>
      <xsl:when test="specdprolog/titleblk/title">      <!-- special division -->
        <xsl:apply-templates select="specdprolog/titleblk/title" mode="navtitle"/>
      </xsl:when>
      <xsl:when test="len">
        <xsl:apply-templates select="len" mode="navtitle"/>
      </xsl:when>
      <xsl:when test="self::msg">
        <xsl:choose>
          <xsl:when test="code"><xsl:apply-templates select="code" mode="navtitle"/></xsl:when>
          <xsl:when test="msgnum and msgtext">
            <xsl:apply-templates select="msgnum" mode="navtitle"/>
            <xsl:text>: </xsl:text>
            <xsl:apply-templates select="msgtext" mode="navtitle"/>
          </xsl:when>
          <xsl:when test="msgnum"><xsl:apply-templates select="msgnum" mode="navtitle"/></xsl:when>
          <xsl:when test="msgtext"><xsl:apply-templates select="msgtext" mode="navtitle"/></xsl:when>
        </xsl:choose>
      </xsl:when>
      <xsl:when test="self::proc and titleblk">
        <xsl:apply-templates select="titleblk" mode="navtitle"/>
      </xsl:when>
      <xsl:when test="self::safety and titleblk">
        <xsl:apply-templates select="titleblk" mode="navtitle"/>
      </xsl:when>
      <xsl:when test="self::ednotices and title">
        <xsl:apply-templates select="title" mode="navtitle"/>
      </xsl:when>
      <xsl:when test="self::modinfo and title">
        <xsl:apply-templates select="title" mode="navtitle"/>
      </xsl:when>
      <xsl:when test="self::partasm and title">
        <xsl:apply-templates select="title" mode="navtitle"/>
      </xsl:when>
      <xsl:when test="self::mod">
        <xsl:apply-templates select="key('moditemdefs',modname/@class)/title" mode="navtitle"/><xsl:text>: </xsl:text>
        <xsl:apply-templates select="modname" mode="navtitle"/>
      </xsl:when>
      <xsl:when test="self::longdesc and title">
        <xsl:apply-templates select="title" mode="navtitle"/>
      </xsl:when>
      <xsl:when test="titleblk">
        <xsl:apply-templates select="titleblk" mode="navtitle"/>
      </xsl:when>
      <xsl:when test="title">
        <xsl:apply-templates select="title" mode="navtitle"/>
      </xsl:when>
    </xsl:choose>
  </xsl:variable>
  <xsl:if test="string-length($navtitle-val)>0">
    <xsl:attribute name="navtitle"><xsl:value-of select="normalize-space($navtitle-val)"/></xsl:attribute>
  </xsl:if>
</xsl:template>
<!-- P016829: pass through all of the text, except that within index terms -->
<xsl:template match="idxterm" mode="navtitle"/>
<xsl:template match="*" mode="navtitle">
  <xsl:apply-templates select="text()|*" mode="navtitle"/>
</xsl:template>
<xsl:template match="text()" mode="navtitle">
  <xsl:value-of select="."/>
</xsl:template>

<!-- Match any elements that do not create chunks, and continue processing children.
     The current depth does not change (still in the current division). -->
<xsl:template match="*" mode="create-map">
  <xsl:param name="current-depth" select="1"/>
  <xsl:param name="maxtoc"/>
  <xsl:choose>
    <xsl:when test="@conloc">
      <!-- Resume processing at the point which was referenced -->
      <xsl:apply-templates select="key('xreflist',@conloc)[1]" mode="create-map">
        <xsl:with-param name="current-depth">
          <xsl:value-of select="$current-depth"/>
        </xsl:with-param>
        <xsl:with-param name="maxtoc"><xsl:value-of select="$maxtoc"/></xsl:with-param>
      </xsl:apply-templates>
    </xsl:when>
    <xsl:otherwise>
      <!-- Continue processing children -->
      <xsl:apply-templates mode="create-map">
        <xsl:with-param name="current-depth" select="$current-depth"/>
        <xsl:with-param name="maxtoc"><xsl:value-of select="$maxtoc"/></xsl:with-param>
      </xsl:apply-templates>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Do not process text nodes when creating the map hierarchy -->
<xsl:template match="text()" mode="create-map"/>

<!-- [1f] Pull index entries from parent containers (such as dlentry) into the first definition -->
<xsl:template name="pull-indices-to-defn">
  <xsl:if test="parent::*/i1|parent::*/i2|parent::*/i3|parent::*/iref">
    <xsl:if test="not(preceding-sibling::defn)">
      <xsl:apply-templates select="parent::*/i1|parent::*/i2|parent::*/i3|parent::*/iref"
                           mode="ensure-proper-placement"/>
    </xsl:if>
  </xsl:if>
</xsl:template>

<!-- [1z] Other general purpose functions -->
<!-- output empty entries to fill out a table row -->
<xsl:template name="add-entries">
  <xsl:param name="remaining" select="1"/>
  <xsl:if test="$remaining > 0">
    <entry/>
    <xsl:call-template name="add-entries">
      <xsl:with-param name="remaining" select="number($remaining)-1"/>
    </xsl:call-template>
  </xsl:if>
</xsl:template>

<!-- Output a default title: map to title element, check for a subtitle -->
<xsl:template name="output-title-as-title">
  <title>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates/>
    <xsl:if test="following-sibling::subtitle">
      <xsl:text>: </xsl:text>
      <xsl:apply-templates select="following-sibling::subtitle"/>
    </xsl:if>
  </title>
</xsl:template>

<!-- ******************** [2] PROLOG and META INFORMATION ***************** -->

<!-- Store classdef info for easy retrieval. This is necessary because classdef
     can appear in the prolog, or dprolog, or specdprolog, so it can take a long
     time to search for it. The key list is used to retrieve outputclass values,
     and style values for phrases. -->
<xsl:key name="classdef" match="classdef" use="@classname"/>

<!-- Elements will be picked out of the prolog in the order that DITA needs them; they
     will not be processed in the IBMIDDoc order. However, Imagedescs and objlib will always
     create chunks, so go ahead and process them now. -->
<xsl:template match="prolog">
  <xsl:apply-templates select="imagedescs|objlib"/>
</xsl:template>

<!-- dintro parents: Abbrev, Abstract, Bibliog, D, Glossary, Legend, MasterIndex, 
                     Part, Preface, SOA 
     Children: Annot, AsmList, Attention, BibList, Bridge, Caution, CGraphic, D, 
               Danger, DL, Fig, FnList, GL, L, Lines, LitData, LQ, MarkList, MkNote, 
               MMObj, ModInfo, Note, NoteList, OL, P, ParmL, PBlk, Screen, Syntax, 
               Table, TitleBlk, TOC, UL, Xmp.                                  -->
<!-- [2a] If DIntro contains nothing but paragraphs, we can map them to long phrases. If it
     contains more complicated markup, give up and place everything in required-cleanup. -->
<xsl:template match="dintro">
  <xsl:choose>
    <xsl:when test="*[not(self::p)]">
      <xsl:call-template name="output-required-cleanup-warning"/>
      <ph><required-cleanup>
        <xsl:apply-templates/>
      </required-cleanup></ph>
    </xsl:when>
    <xsl:otherwise>
      <xsl:apply-templates select="*">
        <xsl:with-param name="parent-element">shortdesc</xsl:with-param>
      </xsl:apply-templates>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- <abstract> outside of the front matter, should be nested in a division or div-like
     element. So, it will map into shortdesc, which only allows phrase-like content.   -->
<xsl:template match="*[not(self::frontm)]/abstract">
  <xsl:apply-templates select="specdprolog"/>
  <xsl:choose>
    <xsl:when test="dbody/*[not(self::p)]">
      <xsl:call-template name="output-required-cleanup-warning"/>
      <ph><required-cleanup>
        <xsl:apply-templates select="dbody/*"/>
      </required-cleanup></ph>
    </xsl:when>
    <xsl:otherwise>
      <xsl:apply-templates select="dbody/*">
        <xsl:with-param name="parent-element">shortdesc</xsl:with-param>
      </xsl:apply-templates>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- [2b] This is called from division-like elements to add <authors> to the prolog.
     It will use the <authors> element from the nearest div-like element. -->
<xsl:template name="output-authors-metadata">
  <xsl:if test="ancestor-or-self::*/dprolog/authors | ancestor-or-self::*/specdprolog/authors">
    <xsl:apply-templates select="(ancestor-or-self::*/dprolog/authors|ancestor-or-self::*/specdprolog/authors)[last()]" mode="gen-metadata"/>
  </xsl:if>
</xsl:template>
<!-- Add authors info to a map, based on the book-level authors tag. -->
<xsl:template name="output-authors-metadata-in-map">
  <xsl:apply-templates select="/ibmiddoc/prolog/ibmbibentry/authors" mode="gen-metadata"/>
</xsl:template>

<xsl:template match="authors"/>   <!-- ignore during fallthrough processing -->
<xsl:template match="authors" mode="gen-metadata">
  <xsl:apply-templates select="author"/>
</xsl:template>

<xsl:template match="authors/desc">
  <xsl:comment><xsl:value-of select="."/></xsl:comment>
</xsl:template>
<xsl:template match="rev/author"/>

<xsl:template match="authors/author">
  <xsl:value-of select="$newline"/>
  <author>
    <xsl:apply-templates mode="output-authors"/>
  </author>
</xsl:template>

<xsl:template match="*" mode="output-authors">
  <xsl:apply-templates mode="output-authors"/>
  <xsl:if test="following-sibling::*">
    <xsl:text> </xsl:text>       <!-- add a space between title, name, address, etc -->
  </xsl:if>
</xsl:template>

<!-- this template will only be reached when parent is authors, not when parent is
    approvers or owners or maintainer -->
<xsl:template match="person">
  <xsl:apply-templates/>
</xsl:template>
<xsl:template match="name">
  <xsl:apply-templates/>
</xsl:template>

<!-- [2c] This is called from division-like elements to add <copyright> to the prolog.
     It will use one of the following, chosen in this order:
     * The copyr referenced by this division, if @copyr is specified
     * The <copyr> element(s) from the nearest div-like element -->
<xsl:template name="output-copyright-metadata">
  <xsl:choose>
    <!-- If a copyright definition is referenced, use that one only -->
    <xsl:when test="@copyr">
      <xsl:apply-templates select="ancestor-or-self::*/dprolog/copyrdefs/copyr[@id=current()/@copyr]|
                                   ancestor-or-self::*/specdprolog/copyrdefs/copyr[@id=current()/@copyr]"/>
      <xsl:apply-templates select="/ibmiddoc/prolog/copyrdefs/copyr[@id=current()/@copyr]"/>
    </xsl:when>
    <xsl:when test="ancestor-or-self::*/dprolog/copyrdefs | ancestor-or-self::*/specdprolog/copyrdefs">
      <xsl:apply-templates select="(ancestor-or-self::*/dprolog/copyrdefs|ancestor-or-self::*/specdprolog/copyrdefs)[last()]" mode="gen-metadata"/>
    </xsl:when>
  </xsl:choose>
</xsl:template>

<xsl:template name="output-copyright-metadata-in-map">
  <xsl:apply-templates select="/ibmiddoc/prolog/copyrdefs" mode="gen-metadata"/>
</xsl:template>

<xsl:template match="copyrdefs"/>   <!-- ignore during fallthrough processing -->
<xsl:template match="copyrdefs" mode="gen-metadata">
  <xsl:apply-templates/>
</xsl:template>

<xsl:template match="copyr">
  <xsl:value-of select="$newline"/>
  <copyright>
    <!-- No way to pull the proper year out of the text? -->
    <copyryear>
      <xsl:attribute name="year"><xsl:value-of select="$YEAR"/></xsl:attribute>
    </copyryear>
    <!-- copyr holds many p-like elements, but copyrholder only takes text -->
    <copyrholder><xsl:value-of select="."/></copyrholder>
  </copyright>
</xsl:template>

<!-- [2d] This is called from division-like elements to add <critdates> to the prolog.
     It will place the nearest div-level <critdate> element into a matching <critdate> element.
     The date contents will be placed in @date, and the desc will be placed in a comment
     so that it will not be lost.  -->
<xsl:template name="output-critdates-metadata">
  <xsl:if test="ancestor-or-self::*/dprolog/critdates | ancestor-or-self::*/specdprolog/critdates">
    <xsl:apply-templates select="(ancestor-or-self::*/dprolog/critdates|ancestor-or-self::*/specdprolog/critdates)[last()]" mode="gen-metadata"/>
  </xsl:if>
</xsl:template>

<!-- If critdates are specified for the entire book, add them to the map -->
<xsl:template name="output-critdates-metadata-in-map">
  <xsl:apply-templates select="/ibmiddoc/prolog/critdates" mode="gen-metadata"/>
</xsl:template>

<xsl:template match="critdates"/>
<xsl:template match="critdates" mode="gen-metadata">
  <xsl:value-of select="$newline"/>
  <critdates>
    <xsl:apply-templates/>
  </critdates>
</xsl:template>

<xsl:template match="critdate">
  <created>
    <xsl:attribute name="date">
      <xsl:value-of select="date"/>
    </xsl:attribute>
  </created>
  <xsl:if test="desc">
    <xsl:comment><xsl:value-of select="desc"/></xsl:comment>
  </xsl:if>
</xsl:template>

<!-- Prevent fallthrough, just in case; content should be pulled by critdate -->
<xsl:template match="critdate/date"/>

<!-- [2e] Add publisher information for the book to the map's metadata  -->
<xsl:template name="output-publisher-metadata-in-map">
  <xsl:apply-templates select="/ibmiddoc/prolog/ibmbibentry/publisher" mode="gen-metadata"/>
</xsl:template>

<xsl:template match="publisher"/>
<xsl:template match="publisher" mode="gen-metadata">
  <!-- publisher only takes text. If address is specfied, it must be reduced to text-only. -->
  <xsl:value-of select="$newline"/>
  <publisher>
    <xsl:apply-templates select="corpname"/>   <!-- corpname only takes text and dvcfobj -->
    <xsl:if test="address">
      <xsl:text>, </xsl:text>
      <xsl:value-of select="address"/>
    </xsl:if>
  </publisher>
</xsl:template>

<xsl:template match="corpname">
  <xsl:apply-templates/>
</xsl:template>

<!-- [2f] This is called from division-like elements to add <permissions> to the prolog.
     It will place the permissions info for this topic into that element's view attribute.
     The permissions contains both classification and, optionally, workflow. -->
<xsl:template name="output-permissions-metadata">
  <xsl:choose>
    <!-- If there is a classification value active -->
    <xsl:when test="ancestor-or-self::*/dprolog/metadata[@classification] |
                    ancestor-or-self::*/specdprolog/metadata[@classification]">
      <xsl:value-of select="$newline"/>
      <permissions>
        <xsl:apply-templates select="(ancestor-or-self::*/dprolog/metadata[@classification] |
                                      ancestor-or-self::*/specdprolog/metadata[@classification])[last()]"
                                      mode="output-permissions"/>
        <!-- If there is also a workflow value currently active -->
        <xsl:if test="ancestor-or-self::*/dprolog/othermeta[@name='workflow'] |
                      ancestor-or-self::*/specdprolog/metadata[@name='workflow']">
          <xsl:apply-templates select="(ancestor-or-self::*/dprolog/othermeta[@name='workflow'] |
                                        ancestor-or-self::*/specdprolog/othermeta[@name='workflow'])[last()]"
                                        mode="output-permissions"/>
        </xsl:if>
      </permissions>
    </xsl:when>
    <!-- If there is no classification, but there is a workflow value -->
    <xsl:when test="ancestor-or-self::*/dprolog/othermeta[@name='workflow'] |
                    ancestor-or-self::*/specdprolog/metadata[@name='workflow']">
      <xsl:value-of select="$newline"/>
      <permissions view="all">     <!-- @view is required, so set it to all -->
        <xsl:apply-templates select="(ancestor-or-self::*/dprolog/othermeta[@name='workflow'] |
                                      ancestor-or-self::*/specdprolog/othermeta[@name='workflow'])[last()]"
                                      mode="output-permissions"/>
      </permissions>
    </xsl:when>
  </xsl:choose>
</xsl:template>

<!-- If this is called, we already know the element has @classification -->
<xsl:template match="metadata" mode="output-permissions">
  <xsl:attribute name="view">
    <xsl:if test="@classification='unclassified'">all</xsl:if>
    <xsl:if test="@classification='classified'">classified</xsl:if>
  </xsl:attribute>
</xsl:template>

<xsl:template match="othermeta" mode="output-permissions">
  <xsl:variable name="workflow" select="@content"/>
  <xsl:if test="not($workflow='author' or $workflow='editor' or $workflow='publisher' or $workflow='reviewer')">
    <!-- output-message for invalid workflow value -->
  </xsl:if>
  <xsl:attribute name="workflow"><xsl:value-of select="@content"/></xsl:attribute>
</xsl:template>

<!-- [2g] Called from div-like elements to add audience information to the metadata. -->
<xsl:template name="output-audience-metadata">
  <xsl:for-each select="ancestor-or-self::*/dprolog/metadata[@job or @experiencelevel or @audience] |
                        ancestor-or-self::*/specdprolog/metadata[@job or @experiencelevel or @audience]">
    <xsl:value-of select="$newline"/>
    <audience>
      <xsl:attribute name="type">
        <xsl:choose>
          <xsl:when test="@audience"><xsl:value-of select="@audience"/></xsl:when>
          <xsl:otherwise>other</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      <xsl:attribute name="job">
        <xsl:choose>
          <xsl:when test="@job='configuring'">other</xsl:when> <!-- output-message: will need to update -->
          <xsl:when test="@job"><xsl:value-of select="@job"/></xsl:when>
          <xsl:otherwise>other</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      <xsl:if test="@job='configuring'">
        <xsl:attribute name="otherjob">configuring</xsl:attribute>
      </xsl:if>
      <xsl:attribute name="experiencelevel">
        <xsl:choose>
          <xsl:when test="@experiencelevel"><xsl:value-of select="@experiencelevel"/></xsl:when>
          <xsl:otherwise>general</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
    </audience>
  </xsl:for-each>
</xsl:template>

<!-- [2h] Called from div-like elements to add category information to the metadata. -->
<xsl:template name="output-category-metadata">
  <xsl:for-each select="ancestor-or-self::*/dprolog/othermeta[@name='category'] |
                        ancestor-or-self::*/specdprolog/othermeta[@name='category']">
    <xsl:value-of select="$newline"/>
    <category><xsl:value-of select="@content"/></category>
  </xsl:for-each>
</xsl:template>

<!-- [2i] Called from div-like elements to add keywords to the metadata. -->
<xsl:template name="output-keywords-metadata">
  <xsl:if test="i1 |i2 | i3 | iref |
                  dprolog//idxterm[not(../parent::title)] | specdprolog//idxterm[not(../parent::title)] |
                  parent::lers/i1 | parent::lers/i2 | parent::lers/i3 | parent::lers/iref |
                  dbody/*[1][name()='i1' or name()='i2' or name()='i3' or name()='iref']">
    <xsl:value-of select="$newline"/>
    <keywords>
      <xsl:apply-templates select="(i1 |i2 | i3 | iref | dprolog//idxterm | specdprolog//idxterm |
                                    parent::lers/i1 | parent::lers/i2 | parent::lers/i3 |
                                    parent::lers/iref)[not(ancestor::i1 or ancestor::i2 or ancestor::i3)]"
                           mode="ensure-proper-placement"/>
      <!-- Any index entries in a dbody AND before any content should be in keywords -->
      <xsl:if test="dbody/i1|dbody/i2|dbody/i3|dbody/iref">
        <xsl:call-template name="add-dbody-indices-to-keywords"/>
      </xsl:if>
      <xsl:value-of select="$newline"/>
    </keywords>
  </xsl:if>
</xsl:template>

<xsl:template name="add-dbody-indices-to-keywords">
  <xsl:for-each select="dbody/i1|dbody/i2|dbody/i3|dbody/iref">
    <xsl:choose>
      <xsl:when test="preceding-sibling::*[not(name()='i1' or name()='i2' or name()='i3' or name()='iref')]">
        <!-- If there is non-index content before this entry, it does not go in keywords -->
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$newline"/>
        <xsl:apply-templates select="." mode="ensure-proper-placement"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:for-each>
</xsl:template>

<!-- [2j] Called from div-like elements to add prodinfo to the metadata. -->
<xsl:template name="output-prodinfo-metadata">
  <xsl:variable name="series">
    <xsl:value-of select="(ancestor-or-self::*/dprolog/othermeta[@name='series']/@content |
                           ancestor-or-self::*/specdprolog/othermeta[@name='series']/@content)[last()]"/>
  </xsl:variable>
  <xsl:for-each select="ancestor-or-self::*/dprolog/ibmprodinfo/prodname |
                        ancestor-or-self::*/specdprolog/ibmprodinfo/prodname |
                        ancestor-or-self::*/dprolog/prodinfo/prodname |
                        ancestor-or-self::*/specdprolog/prodinfo/prodname">
    <!-- Output one prodinfo for each <ibmprodinfo> or <prodinfo> in the hierarchy -->
    <prodinfo>
      <prodname><xsl:apply-templates/></prodname>
      <vrmlist>
        <vrm>
          <xsl:if test="not(following-sibling::version)">
            <xsl:attribute name="version">na</xsl:attribute>
          </xsl:if>
          <!-- Will apply version, release, and modlvl -->
          <xsl:apply-templates select="following-sibling::*" mode="gen-metadata"/>
        </vrm>
      </vrmlist>
      <xsl:if test="ancestor::*/dprolog/othermeta[@name='series']|ancestor::*/specdprolog/othermeta[@name='series']">
        <series>
          <xsl:value-of select="(ancestor::*/dprolog/othermeta[@name='series']/@content |
                                 ancestor::*/dprolog/othermeta[@name='series']/@content)[last()]"/>
        </series>
      </xsl:if>
      <xsl:if test="ancestor::*/dprolog/othermeta[@name='platform']|ancestor::*/specdprolog/othermeta[@name='platform']">
        <platform>
          <xsl:value-of select="(ancestor::*/dprolog/othermeta[@name='platform']/@content |
                                 ancestor::*/dprolog/othermeta[@name='platform']/@content)[last()]"/>
        </platform>
      </xsl:if>
      <xsl:if test="ancestor::*/dprolog/othermeta[@name='component']|ancestor::*/specdprolog/othermeta[@name='component']">
        <component>
          <xsl:value-of select="(ancestor::*/dprolog/othermeta[@name='component']/@content |
                                 ancestor::*/dprolog/othermeta[@name='component']/@content)[last()]"/>
        </component>
      </xsl:if>
    </prodinfo>
  </xsl:for-each>
</xsl:template>

<xsl:template name="output-prodinfo-metadata-in-map">
  <xsl:choose>
    <xsl:when test="/ibmiddoc/prolog/ibmprodinfo/prodname or /ibmiddoc/prolog/prodinfo/prodname">
      <xsl:for-each select="/ibmiddoc/prolog/ibmprodinfo/prodname | /ibmiddoc/prolog/prodinfo/prodname">
        <!-- Output one prodinfo for each book-level <ibmprodinfo> or <prodinfo> tag -->
        <prodinfo>
          <prodname><xsl:apply-templates/></prodname>
          <vrmlist>
            <vrm>
              <xsl:if test="not(following-sibling::version)">
                <xsl:attribute name="version">na</xsl:attribute>
              </xsl:if>
              <!-- Will apply version, release, and modlvl -->
              <xsl:apply-templates select="following-sibling::*" mode="gen-metadata"/>
            </vrm>
          </vrmlist>
          <xsl:if test="/ibmiddoc/@brand">
            <brand><xsl:value-of select="/ibmiddoc/@brand"/></brand>
          </xsl:if>
          <xsl:if test="/ibmiddoc/@newbrand">
            <brand><xsl:value-of select="/ibmiddoc/@newbrand"/></brand>
          </xsl:if>
        </prodinfo>
      </xsl:for-each>
    </xsl:when>
    <!-- If there is a brand, but no prodinfo, save the brand in DITA's brand element -->
    <xsl:when test="/ibmiddoc/@brand|/ibmiddoc/@newbrand">
      <prodinfo>
        <prodname/>
        <vrmlist><vrm version="na"/></vrmlist>
        <xsl:if test="/ibmiddoc/@brand">
          <brand><xsl:value-of select="/ibmiddoc/@brand"/></brand>
        </xsl:if>
        <xsl:if test="/ibmiddoc/@newbrand">
          <brand><xsl:value-of select="/ibmiddoc/@newbrand"/></brand>
        </xsl:if>
      </prodinfo>
    </xsl:when>
  </xsl:choose>
</xsl:template>

<xsl:template match="prodinfo|ibmprodinfo"/>  <!-- Inserted directly in named templates above -->

<xsl:template match="prodname"/>  <!-- The prodname tag is generated in output-prodinfo-metadata -->

<xsl:template match="version"/>
<xsl:template match="version" mode="gen-metadata">
  <xsl:attribute name="version"><xsl:value-of select="."/></xsl:attribute>
</xsl:template>

<xsl:template match="release"/>
<xsl:template match="release" mode="gen-metadata">
  <xsl:attribute name="release"><xsl:value-of select="."/></xsl:attribute>
</xsl:template>

<xsl:template match="modlvl"/>
<xsl:template match="modlvl" mode="gen-metadata">
  <xsl:attribute name="modification"><xsl:value-of select="."/></xsl:attribute>
</xsl:template>

<!-- Content of ibmfeatnum and ibmpgmnum is pulled into othermeta -->
<xsl:template match="ibmfeatnum"/>
<xsl:template match="ibmfeatnum" mode="gen-metadata"/>
<xsl:template match="ibmpgmnum"/>
<xsl:template match="ibmpgmnum" mode="gen-metadata"/>

<!-- [2k] Called from div-like elements to add othermeta to the metadata. This series
     of templates will add anything which needs to go into an othermeta element, even
     if it comes from a unique IBMIDDoc element. -->
<xsl:template name="output-othermeta-metadata">
  <xsl:for-each select="ancestor-or-self::*/dprolog/othermeta | ancestor-or-self::*/specdprolog/othermeta">
    <xsl:choose>
      <!-- Skip othermeta tags that are mapped to specific elements elsewhere -->
      <xsl:when test="@name='permissions' or @name='category' or @name='series' or
                      @name='platform' or @name='component'">
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$newline"/>
        <othermeta>
          <xsl:attribute name="name"><xsl:value-of select="@name"/></xsl:attribute>
          <xsl:attribute name="content"><xsl:value-of select="@content"/></xsl:attribute>
        </othermeta>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:for-each>
</xsl:template>

<xsl:template name="output-vrmoriginated-as-othermeta">
  <xsl:for-each select="ancestor-or-self::*/dprolog/metadata/@vrmoriginated |
                        ancestor-or-self::*/specdprolog/metadata/@vrmoriginated">
    <xsl:value-of select="$newline"/>
    <othermeta name="vrmoriginated">
      <xsl:attribute name="content"><xsl:value-of select="."/></xsl:attribute>
    </othermeta>
  </xsl:for-each>
</xsl:template>

<xsl:template name="output-vrmlastchanged-as-othermeta">
  <xsl:for-each select="ancestor-or-self::*/dprolog/metadata/@vrmlastchanged |
                        ancestor-or-self::*/specdprolog/metadata/@vrmlastchanged">
    <xsl:value-of select="$newline"/>
    <othermeta name="vrmlastchanged">
      <xsl:attribute name="content"><xsl:value-of select="."/></xsl:attribute>
    </othermeta>
  </xsl:for-each>
</xsl:template>

<xsl:template name="output-otherprops-as-othermeta">
  <xsl:if test="@otherprops">
    <xsl:value-of select="$newline"/>
    <othermeta name="otherprops">
      <xsl:attribute name="content"><xsl:value-of select="@otherprops"/></xsl:attribute>
    </othermeta>
  </xsl:if>
</xsl:template>

<xsl:template name="output-ibmpgmnum-as-othermeta">
  <xsl:for-each select="ancestor-or-self::*/dprolog/ibmprodinfo/ibmpgmnum |
                        ancestor-or-self::*/specdprolog/ibmpgmnum">
    <xsl:value-of select="$newline"/>
    <othermeta name="ibmpgmnum">
      <xsl:attribute name="content"><xsl:value-of select="."/></xsl:attribute>
    </othermeta>
  </xsl:for-each>
</xsl:template>

<xsl:template name="output-ibmfeatnum-as-othermeta">
  <xsl:for-each select="ancestor-or-self::*/dprolog/ibmprodinfo/ibmfeatnum |
                        ancestor-or-self::*/specdprolog/ibmfeatnum">
    <xsl:value-of select="$newline"/>
    <othermeta name="ibmfeatnum">
      <xsl:attribute name="content"><xsl:value-of select="."/></xsl:attribute>
    </othermeta>
  </xsl:for-each>
</xsl:template>

<xsl:template name="output-ibmpgmnum-as-othermeta-in-map">
  <xsl:for-each select="/ibmiddoc/prolog/ibmprodinfo/ibmpgmnum">
    <xsl:value-of select="$newline"/>
    <othermeta name="ibmpgmnum">
      <xsl:attribute name="content"><xsl:value-of select="."/></xsl:attribute>
    </othermeta>
  </xsl:for-each>
</xsl:template>

<xsl:template name="output-ibmfeatnum-as-othermeta-in-map">
  <xsl:for-each select="/ibmiddoc/prolog/ibmprodinfo/ibmfeatnum">
    <xsl:value-of select="$newline"/>
    <othermeta name="ibmfeatnum">
      <xsl:attribute name="content"><xsl:value-of select="."/></xsl:attribute>
    </othermeta>
  </xsl:for-each>
</xsl:template>

<!-- There will only be one of each of these, but using for-each means we do not have to re-select it -->
<xsl:template name="output-filenum-as-othermeta-in-map">
  <xsl:for-each select="/ibmiddoc/prolog/ibmbibentry/filenum">
    <xsl:value-of select="$newline"/>
    <othermeta name="filenum">
      <xsl:attribute name="content"><xsl:value-of select="."/></xsl:attribute>
    </othermeta>
  </xsl:for-each>
</xsl:template>
<xsl:template name="output-ibmpartnum-as-othermeta-in-map">
  <xsl:for-each select="/ibmiddoc/prolog/ibmbibentry/ibmpartnum">
    <xsl:value-of select="$newline"/>
    <othermeta name="ibmpartnum">
      <xsl:attribute name="content"><xsl:value-of select="."/></xsl:attribute>
    </othermeta>
  </xsl:for-each>
</xsl:template>
<xsl:template name="output-ibmdocnum-as-othermeta-in-map">
  <xsl:for-each select="/ibmiddoc/prolog/ibmbibentry/ibmdocnum">
    <xsl:value-of select="$newline"/>
    <othermeta name="ibmdocnum">
      <xsl:attribute name="content"><xsl:value-of select="."/></xsl:attribute>
    </othermeta>
  </xsl:for-each>
</xsl:template>
<xsl:template name="output-origibmdocnum-as-othermeta-in-map">
  <xsl:for-each select="/ibmiddoc/prolog/ibmbibentry/origibmdocnum">
    <xsl:value-of select="$newline"/>
    <othermeta name="origibmdocnum">
      <xsl:attribute name="content"><xsl:value-of select="."/></xsl:attribute>
    </othermeta>
  </xsl:for-each>
</xsl:template>
<xsl:template name="output-prtloc-as-othermeta-in-map">
  <xsl:for-each select="/ibmiddoc/prolog/ibmbibentry/prtloc">
    <xsl:value-of select="$newline"/>
    <othermeta name="prtloc">
      <xsl:attribute name="content"><xsl:value-of select="."/></xsl:attribute>
    </othermeta>
  </xsl:for-each>
</xsl:template>
<xsl:template name="output-publicid-as-othermeta-in-map">
  <xsl:for-each select="/ibmiddoc/prolog/ibmbibentry/publicid">
    <xsl:value-of select="$newline"/>
    <othermeta name="publicid">
      <xsl:attribute name="content"><xsl:value-of select="."/></xsl:attribute>
    </othermeta>
  </xsl:for-each>
</xsl:template>
<xsl:template name="output-volid-as-othermeta-in-map">
  <xsl:for-each select="/ibmiddoc/prolog/ibmbibentry/volid">
    <xsl:value-of select="$newline"/>
    <othermeta name="volid">
      <xsl:attribute name="content"><xsl:value-of select="."/></xsl:attribute>
    </othermeta>
  </xsl:for-each>
</xsl:template>

<!-- [2z] Empty templates, to prevent fallthrough for various prolog elements. -->
<xsl:template match="approvers"/>
<xsl:template match="owners"/>
<xsl:template match="maintainer"/>
<xsl:template match="prtloc"/>
<xsl:template match="publicid"/>
<xsl:template match="volid"/>

<xsl:template match="sem"/>
<xsl:template match="desc"/>

<xsl:template match="propdefs"/>
<xsl:template match="ldescs | nmlist"/>
<xsl:template match="idxdefs"/>
<xsl:template match="bibentrydefs"/>
<xsl:template match="gldefs"/>
<xsl:template match="revdefs | rev | mark"/>
<xsl:template match="qualifdefs | qualif"/>
<xsl:template match="classdef"/>
<xsl:template match="propdef | propdesc | propgroup"/>
<xsl:template match="mkdesc | mkaction | mkclass"/>
<xsl:template match="containeddocs"/>
<xsl:template match="msgitemdef"/>
<xsl:template match="lersdef"/>
<xsl:template match="oldef | uldef | dldef | gldef | figdef | syntaxdef | screendef"/>

<!-- ******************** [3] DIVISION-LEVEL ELEMENTS ********************** -->
<xsl:template match="ibmiddoc">
  <xsl:apply-templates/>
</xsl:template>

<!-- Most prolog content is only referenced when needed, but imagedescs must 
     be turned in to topics. -->
<xsl:template match="dprolog|specdprolog">
  <!-- output-message for objlib in dprolog? -->
  <xsl:apply-templates select="imagedescs"/>
</xsl:template>

<!-- If @conloc is specified, the topics were all created in the objlib. -->
<xsl:template match="dblk">
  <xsl:apply-templates/>
</xsl:template>

<xsl:template match="appendix">
  <xsl:apply-templates/>
</xsl:template>

<xsl:template match="frontm | body | backm">
  <xsl:apply-templates/>
</xsl:template>

<xsl:template match="backcover | frontcover | coverdef"/>

<xsl:template match="part">
  <xsl:call-template name="output-divlike"/>
</xsl:template>

<xsl:template match="d">
  <xsl:call-template name="output-divlike"/>
</xsl:template>

<!-- Catch division titles, and output any title and subtitle -->
<xsl:template match="d/dprolog/titleblk/title">
  <xsl:call-template name="output-title-as-title"/>
</xsl:template>

<!-- If the division is a concept/reference/task, $body-type will be set to the
     specific element needed (conbody, etc). Otherwise, use body. If there is a
     dsum element after the dbody, output it inside the topic's body. -->
<xsl:template match="dbody">
  <xsl:param name="body-type" select="body"/>
  <xsl:value-of select="$newline"/>
  <xsl:element name="{$body-type}">
    <xsl:apply-templates>
      <xsl:with-param name="parent-element"><xsl:value-of select="$body-type"/></xsl:with-param>
    </xsl:apply-templates>
    <xsl:apply-templates select="following-sibling::dsum">
      <xsl:with-param name="parent-element"><xsl:value-of select="$body-type"/></xsl:with-param>
    </xsl:apply-templates>
    <xsl:value-of select="$newline"/>
  </xsl:element>
</xsl:template>

<!-- Dsum only exists in MasterIndex, or in elements that also require a dbody.
     MasterIndex is suppressed, so it does not matter. All other elements will have
     the dsum template called from dbody. 
     All current body types accept <section> as the last element, so we do not yet
     have to make use of the body-type parameter. -->
<xsl:template match="dsum">
  <xsl:param name="body-type">body</xsl:param>
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <xsl:when test="$parent-element='result'">
      <xsl:if test="titleblk">
        <xsl:call-template name="output-required-cleanup-warning"/>
        <required-cleanup><xsl:apply-templates select="titleblk"/></required-cleanup>
      </xsl:if>
      <xsl:apply-templates select="*[not(self::titleblk)]">
        <xsl:with-param name="parent-element">result</xsl:with-param>
      </xsl:apply-templates>
    </xsl:when>
    <xsl:otherwise>
      <xsl:value-of select="$newline"/>
      <section>
        <xsl:call-template name="add-default-attributes"/>
        <xsl:apply-templates>
          <xsl:with-param name="parent-element">section</xsl:with-param>
        </xsl:apply-templates>
      </section>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Div-like elements specific to front matter and back matter -->
<xsl:template match="abbrev | bibliog | glossary | legend | preface | safety | soa | 
                     frontm/abstract | ednotices | notices">
  <xsl:call-template name="output-divlike"/>
</xsl:template>

<xsl:template match="ibmsafety">
  <xsl:call-template name="unsupported-element"/>
</xsl:template>

<!-- E003599: Add support for CLE. This includes the following CLE related templates,
     as well as the updates for @spec=man in this template. -->
<xsl:template match="toc">
  <xsl:choose>
    <xsl:when test="parent::frontm">
      <!-- Already generating a map for the entire document. If there is a manual
           TOC, that should be placed in a new map. -->
        <xsl:if test="@spec='man'">
          <xsl:variable name="mapname">
            <xsl:choose>
              <xsl:when test="@chunk-filename"><xsl:value-of select="@chunk-filename"/></xsl:when>
              <xsl:otherwise><xsl:value-of select="$BASEFILENAME"/>-cle.ditamap</xsl:otherwise>
            </xsl:choose>
          </xsl:variable>
          <saxon:output href="{$mapname}">
            <xsl:call-template name="use-map-dtd"/>
            <map>
              <xsl:if test="titleblk/title">
                <xsl:attribute name="title"><xsl:value-of select="normalize-space(titleblk/title)"/></xsl:attribute>
              </xsl:if>
              <xsl:call-template name="generate-metadata-in-map"/>
              <xsl:apply-templates select="cle[1]"/>
            </map>
          </saxon:output>
        </xsl:if>
    </xsl:when>
    <xsl:otherwise>
      <!-- Eventually - might be able to support PTOC -->
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- When in the default template, we know the CLE will create a new topicref. -->
<xsl:template match="cle">
  <xsl:param name="parent-level">0</xsl:param>
  <xsl:param name="current-level"><xsl:value-of select="number($parent-level)+1"/></xsl:param>
  <topicref>
    <!-- If this points to an ID, create an HREF to that file -->
    <xsl:if test="@refid and key('xreflist',@refid)">
      <xsl:attribute name="href"><xsl:value-of select="key('xreflist',@refid)[1]/@chunk-filename"/></xsl:attribute>
    </xsl:if>
    <!-- Create a title.
         If there is text specified locally, add a navtitle, and set locktitle.
         If there is no text but there is a refid, get the title from the target div.
         Otherwise, this CLE is an error. -->
    <xsl:choose>
      <xsl:when test="text()|*">
        <xsl:attribute name="navtitle"><xsl:value-of select="normalize-space(.)"/></xsl:attribute>
        <xsl:attribute name="locktitle">yes</xsl:attribute>
      </xsl:when>
      <xsl:when test="@refid and key('xreflist',@refid)">
        <xsl:attribute name="navtitle">
          <xsl:value-of select="key('xreflist',@refid)[1]/dprolog/titleblk/title|
                                key('xreflist',@refid)[1]/specdprolog/titleblk/title"/>
        </xsl:attribute>
      </xsl:when>
    </xsl:choose>
    <!-- Find children of this topic. Those are following topics that are nested lower,
         which appear before another CLE with an equal or higher level. -->
    <xsl:apply-templates select="following-sibling::cle[1]" mode="find-nested-topic">
      <xsl:with-param name="previous-level" select="$current-level"/>
    </xsl:apply-templates>
  </topicref>
    <!-- Find following siblings that are at this level. If there are higher-level
         divisions before the next one at this level, the next one at this level will
         not be reached; it will be contained within the higher level division. -->
  <xsl:apply-templates select="following-sibling::cle[1]" mode="find-sibling-topic">
    <xsl:with-param name="match-level" select="$current-level"/>
  </xsl:apply-templates>
</xsl:template>

<!-- Find the level of the current CLE. If @refid is specified, the level is the level
     of the target division. Otherwise, if @lvl is specified, use it. Otherwise, 1. -->
<xsl:template name="find-cle-level">
  <xsl:choose>
    <xsl:when test="@refid and key('xreflist',@refid)">
      <xsl:value-of select="count((key('xreflist',@refid))[1]/ancestor-or-self::*[@chunk-filename])"/>
    </xsl:when>
    <xsl:when test="@lvl"><xsl:value-of select="@lvl"/></xsl:when>
    <xsl:otherwise>1</xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Find a sibling to the CLE we just closed. 
     - If the current CLE is deeper (the current level is higher than the match level),
       then the one we just closed contained this CLE; move on to check the next one.
     - If the current CLE is at the same level, this is a sibling, so process it.
     - Otherwise, this CLE is not a sibling or a child, so stop looking: it is a
       sibling of a higher level CLE, so pop back up and keep checking.      -->
<xsl:template match="cle" mode="find-sibling-topic">
  <xsl:param name="match-level"/>
  <xsl:param name="current-level">
    <xsl:call-template name="find-cle-level"/>
  </xsl:param>
  <xsl:choose>
    <!-- When this CLE matches the level of the previous one, output it as a sibling -->
    <xsl:when test="$current-level=$match-level">
      <xsl:apply-templates select=".">
        <xsl:with-param name="parent-level"><xsl:value-of select="number($match-level)-1"/></xsl:with-param>
      </xsl:apply-templates>
    </xsl:when>
    <!-- This CLE was contained by the previous one, so move on to the next one -->
    <xsl:when test="$current-level &gt; $match-level">
      <xsl:apply-templates select="following-sibling::cle[1]" mode="find-sibling-topic">
        <xsl:with-param name="match-level" select="$match-level"/>
      </xsl:apply-templates>
    </xsl:when>
    <!-- Otherwise, there are no more siblings -->
  </xsl:choose>
</xsl:template>

<!-- Find children of a CLE. If the current CLE is deeper than the test CLE, then this
     one is a child. So, process it. If there are further children that are siblings,
     they will be added using the find-sibling-topic mode when this CLE is processed. -->
<xsl:template match="cle" mode="find-nested-topic">
  <xsl:param name="previous-level"/>
  <xsl:param name="current-level">
    <xsl:call-template name="find-cle-level"/>
  </xsl:param>
  <xsl:if test="$current-level &gt; $previous-level">
    <!-- This one is a child, so add it to the open topicref -->
    <xsl:apply-templates select=".">
      <xsl:with-param name="parent-level" select="$previous-level"/>
    </xsl:apply-templates>
  </xsl:if>
</xsl:template>

<!-- Front or back matter elements that cannot map to DITA -->
<xsl:template match="tlist | figlist | rcf | index | pnindex | masterindex | 
                     masterindexinfo | masterindexobj | masterindexprefix"/>

<!-- [3a] Output the title for a topic, reftopic, task, concept. Depending on what
          div-like element is being evaluated, the title could be in several
          different locations. -->
<xsl:template name="output-topic-title">
  <xsl:choose>
    <xsl:when test="dprolog/titleblk/title">          <!-- standard division -->
      <xsl:apply-templates select="dprolog/titleblk/title"/>
    </xsl:when>
    <xsl:when test="specdprolog/titleblk/title">      <!-- special division -->
      <xsl:apply-templates select="specdprolog/titleblk/title"/>
    </xsl:when>
    <xsl:when test="len">
      <xsl:apply-templates select="len"/>
    </xsl:when>
    <xsl:when test="self::safety and titleblk">
      <xsl:apply-templates select="titleblk"/>
    </xsl:when>
    <xsl:when test="self::ednotices and title">
      <xsl:apply-templates select="title"/>
    </xsl:when>
    <xsl:when test="self::modinfo and title">
      <xsl:apply-templates select="title"/>
    </xsl:when>
    <xsl:when test="self::partasm and title">
      <xsl:apply-templates select="title"/>
    </xsl:when>
    <xsl:when test="self::longdesc and title">
      <xsl:apply-templates select="title"/>
    </xsl:when>
    <xsl:when test="self::smalldoc and smalldocprolog/masthead/mastpart">
      <xsl:apply-templates select="smalldocprolog/masthead"/>
    </xsl:when>
    <xsl:when test="self::notices">
      <title>&nbsp;</title>
    </xsl:when>
    <xsl:otherwise>
      <title>&nbsp;</title>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- [3b] Output the titlealts for a topic. These should generally be in the same
          locations for all div-like elements (unless the element does not have
          an equivalent) -->
<xsl:template name="output-topic-titlealts">
  <xsl:if test="dprolog/titleblk/stitle | specdprolog/titleblk/stitle | titleblk/stitle |
                retkey | dprolog/retkey">
    <xsl:value-of select="$newline"/>
    <titlealts>
      <xsl:if test="dprolog/titleblk/stitle | specdprolog/titleblk/stitle | titleblk/stitle">
        <navtitle>
          <xsl:apply-templates select="./dprolog/titleblk/stitle | ./specdprolog/titleblk/stitle"/>
        </navtitle>
      </xsl:if>
      <xsl:if test="retkey | dprolog/retkey">
        <!-- Only one is possible, so this will not be merging 2 values without a space -->
        <xsl:variable name="retkeyval"><xsl:value-of select="retkey | dprolog/retkey"/></xsl:variable>
        <searchtitle>
          <xsl:value-of select="$retkeyval"/>
        </searchtitle>
      </xsl:if>
    </titlealts>
  </xsl:if>
</xsl:template>

<!-- [3c] Output the shortdesc for a topic, reftopic, task, concept -->
<xsl:template name="output-topic-shortdesc">
  <xsl:if test="dintro | abstract">
    <!-- Contents of these elements are blocks, but shortdesc only allows phrases. -->
    <shortdesc><xsl:apply-templates select="dintro|abstract"/></shortdesc>
  </xsl:if>
  <xsl:if test="self::modinfo and child::desc">
    <xsl:apply-templates select="desc"/>
  </xsl:if>
</xsl:template>

<!-- [3d] Output a prolog in a topic. Always output the <prolog> tag, because
     it is nearly always needed, and it is too complex to check every case that
     is needed to add it. Every individual meta element is handled in a named
     template, which only creates the meta tags if they are needed. -->
<xsl:template name="output-topic-prolog">
  <xsl:value-of select="$newline"/>
  <prolog>
    
    <xsl:call-template name="output-authors-metadata"/>
    <!-- nothing maps to the <source> element; if it did, it would go here -->
    <!-- publisher goes here; it is only book-level, so only place it in the map -->
    <xsl:call-template name="output-copyright-metadata"/>
    <xsl:call-template name="output-critdates-metadata"/>
    <xsl:call-template name="output-permissions-metadata"/>
    
    <!-- output metadata element if we need any of the nested tags:
         * test for audience in a dprolog or specdprolog, or for metadata vrm attributes
         * test for category in a dprolog or specdprolog
         * test for prodinfo, ibmprodinfo in dprolog, or specdprolog 
         * test for othermeta in a dprolog or specdprolog
         * test for index entries that apply to this entire topic
         * test for a props value on this topic -->
    <xsl:if test="ancestor-or-self::*/dprolog/metadata[@job or @experiencelevel or @audience or @vrmoriginated or @vrmlastchanged] | 
                  ancestor-or-self::*/specdprolog/metadata[@job or @experiencelevel or @audience or @vrmoriginated or @vrmlastchanged] |
                  ancestor-or-self::*/dprolog/othermeta[@name='category'] |
                  ancestor-or-self::*/specdprolog/othermeta[@name='category'] |
                  ancestor-or-self::*/dprolog/prodinfo | ancestor-or-self::*/dprolog/ibmprodinfo | 
                  ancestor-or-self::*/specdprolog/prodinfo | ancestor-or-self::*/specdprolog/ibmprodinfo |
                  ancestor-or-self::*/dprolog/othermeta | ancestor-or-self::*/specdprolog/othermeta |
                  i1 | i2 | i3 | iref | 
                  dprolog//idxterm[not(../parent::title)] | specdprolog//idxterm[not(../parent::title)] |
                  parent::lers/i1 | parent::lers/i2 | parent::lers/i3 | parent::lers/iref |
                  dbody/*[1][name()='i1' or name()='i2' or name()='i3' or name()='iref']">
      <metadata>
        <xsl:call-template name="output-audience-metadata"/>
        <xsl:call-template name="output-category-metadata"/>
        <xsl:call-template name="output-keywords-metadata"/>
        <xsl:call-template name="output-prodinfo-metadata"/>  <!-- create prodinfo with prodname, vrmlist -->
        <xsl:call-template name="output-othermeta-metadata"/>
        <!-- output IBMIDDoc elements that map to othermeta in DITA -->
        <xsl:call-template name="output-vrmoriginated-as-othermeta"/>
        <xsl:call-template name="output-vrmlastchanged-as-othermeta"/>
        <!--<xsl:call-template name="output-otherprops-as-othermeta"/>-->
        <xsl:call-template name="output-ibmpgmnum-as-othermeta"/>
        <xsl:call-template name="output-ibmfeatnum-as-othermeta"/>
        <xsl:value-of select="$newline"/>
      </metadata>
    </xsl:if>
    
    <!-- output resourceid? -->
    <xsl:value-of select="$newline"/>
  </prolog>
</xsl:template>

<!-- [3e] Output the body of the topic. Any nested topic-level elements which are NOT chunked
     should not be processed. If they are, there will be topics nested in a body. They
     should instead be placed after the body. Topic-level elements which are chunked can
     be processed inline or after the body; when they are processed does not matter, because
     they will be directed to other files. 
     Note: currently all topic level elements are chunked, so this is not a concern. 
     Many elements have titles as children; these were placed in the topic title, and should
     not be re-processed.-->
<xsl:template name="output-topic-body">
  <xsl:param name="infotype">topic</xsl:param>
  <xsl:param name="body-type">body</xsl:param>
  <xsl:choose>
    <xsl:when test="self::le">
      <xsl:element name="{$body-type}">
        <xsl:apply-templates select="*[not(self::len or self::retkey)]">
          <xsl:with-param name="parent-element"><xsl:value-of select="$body-type"/></xsl:with-param>
        </xsl:apply-templates>
      </xsl:element>
    </xsl:when>
    <!-- No div-like children directly in notices or ednotices -->
    <xsl:when test="self::ednotices or self::notices">
      <xsl:element name="{$body-type}">
        <xsl:apply-templates select="*[not(self::title)]">
          <xsl:with-param name="parent-element"><xsl:value-of select="$body-type"/></xsl:with-param>
        </xsl:apply-templates>
      </xsl:element>
    </xsl:when>
    <xsl:when test="self::partasm">
      <xsl:element name="{$body-type}">
        <xsl:apply-templates select="*[not(self::title) and not(self::retkey)]">
          <xsl:with-param name="parent-element"><xsl:value-of select="$body-type"/></xsl:with-param>
        </xsl:apply-templates>
      </xsl:element>
    </xsl:when>
    <xsl:when test="self::safety">
      <xsl:element name="{$body-type}">
        <xsl:apply-templates select="*[not(self::titleblk)]">
          <xsl:with-param name="parent-element"><xsl:value-of select="$body-type"/></xsl:with-param>
        </xsl:apply-templates>
      </xsl:element>
    </xsl:when>
    <xsl:when test="self::modinfo">
      <xsl:element name="{$body-type}">
        <xsl:apply-templates select="*[not(self::title) and not(self::retkey) and not(self::desc)]">
          <xsl:with-param name="parent-element"><xsl:value-of select="$body-type"/></xsl:with-param>
        </xsl:apply-templates>
      </xsl:element>
    </xsl:when>
    <xsl:when test="self::longdesc">
      <xsl:element name="{$body-type}">
        <xsl:apply-templates select="*[not(self::title)]">
          <xsl:with-param name="parent-element"><xsl:value-of select="$body-type"/></xsl:with-param>
        </xsl:apply-templates>
      </xsl:element>
    </xsl:when>
    <xsl:otherwise>
      <xsl:apply-templates select="dbody">
        <xsl:with-param name="body-type" select="$body-type"/>
      </xsl:apply-templates>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- [3e.1] Functions dealing specifically with creating a <taskbody>. -->

<!-- If an element maps to <task> and has a nested ordered list, the first ordered list will
     be mapped to <steps>. Everything before the list will be placed in <context>, and 
     anything after the steps will be placed in <postreq>.
     E003199: MP advises mapping to result instead of postreq. -->
<xsl:template name="output-task-body-with-steps">
  <!-- Save the position of the first OL, so that it will not have to be recalculated every
       time an element position is checked. -->
  <xsl:variable name="olposition">
    <xsl:choose>
      <xsl:when test="dbody">
        <xsl:for-each select="dbody/ol[1]"><xsl:value-of select="count(preceding-sibling::*)+1"/></xsl:for-each>
      </xsl:when>
      <xsl:otherwise>
        <xsl:for-each select="ol[1]"><xsl:value-of select="count(preceding-sibling::*)+1"/></xsl:for-each>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:choose>
    <!-- The following divlike elements were removed from this test, because they cannot contain
         <ol> elements: LE, PartASM, ModInfo, longdesc -->
    <xsl:when test="self::ednotices or self::notices">
      <context>
        <xsl:apply-templates select="*[not(self::title)][position()&lt;number($olposition)]">
          <xsl:with-param name="parent-element">context</xsl:with-param>
        </xsl:apply-templates>
      </context>
      <xsl:apply-templates select="ol[1]" mode="output-as-steps">
        <xsl:with-param name="parent-element">taskbody</xsl:with-param>
      </xsl:apply-templates>
      <result>
        <xsl:apply-templates select="*[not(self::title)][position()&gt;number($olposition)]">
          <xsl:with-param name="parent-element">result</xsl:with-param>
        </xsl:apply-templates>
      </result>
    </xsl:when>
    <xsl:when test="self::safety">
      <context>
        <xsl:apply-templates select="*[not(self::titleblk)][position()&lt;number($olposition)]">
          <xsl:with-param name="parent-element">context</xsl:with-param>
        </xsl:apply-templates>
      </context>
      <xsl:apply-templates select="ol[1]" mode="output-as-steps">
        <xsl:with-param name="parent-element">taskbody</xsl:with-param>
      </xsl:apply-templates>
      <result>
        <xsl:apply-templates select="*[not(self::titleblk)][position()&gt;number($olposition)]">
          <xsl:with-param name="parent-element">result</xsl:with-param>
        </xsl:apply-templates>
      </result>
    </xsl:when>
    <xsl:otherwise>
      <context>
        <xsl:apply-templates select="dbody/*[position()&lt;number($olposition)]">
          <xsl:with-param name="parent-element">context</xsl:with-param>
        </xsl:apply-templates>
      </context>
      <xsl:apply-templates select="dbody/ol[1]" mode="output-as-steps">
        <xsl:with-param name="parent-element">taskbody</xsl:with-param>
      </xsl:apply-templates>
      <result>
        <xsl:apply-templates select="dbody/*[position()&gt;number($olposition)]">
          <xsl:with-param name="parent-element">result</xsl:with-param>
        </xsl:apply-templates>
        <xsl:apply-templates select="dsum">
          <xsl:with-param name="parent-element">result</xsl:with-param>
        </xsl:apply-templates>
      </result>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- If a division maps to task but does not contain an ordered list, place
     everything in <context>. If processing dbody and it is followed by
     dsum, place the dsum in postreq.
     E003199: place dsum in result instead of postreq. -->
<xsl:template name="output-task-body-without-steps">
  <context>
  <xsl:choose>
    <xsl:when test="self::le">
      <xsl:apply-templates select="*[not(name()='len')]">
        <xsl:with-param name="parent-element">context</xsl:with-param>
      </xsl:apply-templates>
    </xsl:when>
    <!-- No div-like children directly in notices or ednotices -->
    <xsl:when test="self::ednotices or self::notices">
      <xsl:apply-templates select="*[not(name()='title')]">
        <xsl:with-param name="parent-element">context</xsl:with-param>
      </xsl:apply-templates>
    </xsl:when>
    <xsl:when test="self::partasm">
      <xsl:apply-templates select="*[not(name()='title') and not(name()='retkey')]">
        <xsl:with-param name="parent-element">context</xsl:with-param>
      </xsl:apply-templates>
    </xsl:when>
    <xsl:when test="self::safety">
      <xsl:apply-templates select="*[not(name()='titleblk')]">
        <xsl:with-param name="parent-element">context</xsl:with-param>
      </xsl:apply-templates>
    </xsl:when>
    <xsl:when test="self::modinfo">
      <xsl:apply-templates select="*[not(name()='title') and not(name()='retkey') and not(name()='desc')]">
        <xsl:with-param name="parent-element">context</xsl:with-param>
      </xsl:apply-templates>
    </xsl:when>
    <xsl:when test="self::longdesc">
      <xsl:apply-templates select="*[not(name()='title')]">
        <xsl:with-param name="parent-element">context</xsl:with-param>
      </xsl:apply-templates>
    </xsl:when>
    <xsl:otherwise>
      <xsl:apply-templates select="dbody/*">
        <xsl:with-param name="body-type">taskbody</xsl:with-param>
        <xsl:with-param name="parent-element">context</xsl:with-param>
      </xsl:apply-templates>
    </xsl:otherwise>
  </xsl:choose>
  </context>
  <xsl:if test="dsum">
    <result>
      <xsl:apply-templates select="dsum">
        <xsl:with-param name="parent-element">result</xsl:with-param>
      </xsl:apply-templates>
    </result>
  </xsl:if>
</xsl:template>

<!-- Output a task body. Decides whether the task includes steps, and calls
     the appropriate named template to output the task body. -->
<xsl:template name="output-task-body">
  <taskbody>
    <xsl:choose>
      <xsl:when test="ol|dbody/ol">
        <xsl:call-template name="output-task-body-with-steps"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:call-template name="output-task-body-without-steps"/>
      </xsl:otherwise>
    </xsl:choose>
  </taskbody>
</xsl:template>

<!-- [3f] Output the contents of a topic. Call this template after
          the initial <topic>, <concept>, etc tag is created. The infotype should
          be passed in as a parameter, and is used to determine the body type. -->
<xsl:template name="generate-topic-content">
  <xsl:param name="infotype" select="topic"/>
  <xsl:variable name="body-type">
    <xsl:choose>
      <xsl:when test="$infotype='concept'">conbody</xsl:when>
      <xsl:when test="$infotype='reference'">refbody</xsl:when>
      <xsl:when test="$infotype='task'">taskbody</xsl:when>
      <xsl:otherwise>body</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:call-template name="add-default-attributes"/>
  
  <xsl:value-of select="$newline"/>
  <xsl:call-template name="output-topic-title"/>
  <xsl:call-template name="output-topic-titlealts"/>
  <xsl:call-template name="output-topic-shortdesc"/>
  <xsl:if test="not(self::longdesc)">  <!-- Longdesc cannot have a prolog -->
    <xsl:call-template name="output-topic-prolog"/>
  </xsl:if>
  <!-- Call a template to output the body. Task has special handling; all others use the default. -->
  <xsl:choose>
    <xsl:when test="$infotype='task'">
      <xsl:call-template name="output-task-body"/>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="output-topic-body">
        <xsl:with-param name="body-type"><xsl:value-of select="$body-type"/></xsl:with-param>
      </xsl:call-template>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- These 3 templates can be easily updated or overridden to change the default 
     behavior for creating a task, concept, or reference. Currenly, all use the
     same templates as task. The root element has already been created at this point. -->
<xsl:template name="generate-concept-content">
  <xsl:call-template name="generate-topic-content">
    <xsl:with-param name="infotype">concept</xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template name="generate-reference-content">
  <xsl:call-template name="generate-topic-content">
    <xsl:with-param name="infotype">reference</xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template name="generate-task-content">
  <xsl:call-template name="generate-topic-content">
    <xsl:with-param name="infotype">task</xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!-- [3g] Output a div-like element as a topic. Currently all div-like elements should
          have @chunk-filename specified, and will map to a file. -->
<xsl:template name="output-divlike">
  <xsl:param name="default-type">topic</xsl:param>
  <xsl:variable name="infotype">
    <xsl:call-template name="determine-topic-type">
      <xsl:with-param name="default-type" select="$default-type"/>
    </xsl:call-template>
  </xsl:variable>

  <!-- If @conloc is specified, the division was mapped to a topic when "objlib" was processed. -->
  <xsl:if test="not(@conloc)">
    <!-- Process the dprolog or specdprolog, if present; all that should do is
         process imagedescs, so we will not get extra output -->
    <xsl:apply-templates select="dprolog|specdprolog"/>
    
    <xsl:choose>
      <!-- SAXON is the only supported output processor. Any others will send all output directly
           to the output stream; the file will have to be chunked manually, and DOCTYPEs will have
           to be updated by hand. -->
      <xsl:when test="@chunk-filename and contains(system-property('xsl:vendor'), 'SAXON')">
        <saxon:output href="{@chunk-filename}">
          <xsl:choose>
            <xsl:when test="$infotype='concept'">
              <xsl:call-template name="use-concept-dtd"/>
              <concept><xsl:call-template name="add-language-attribute"/><xsl:call-template name="generate-concept-content"/></concept>
            </xsl:when>
            <xsl:when test="$infotype='task'">
              <xsl:call-template name="use-task-dtd"/>
              <task><xsl:call-template name="add-language-attribute"/><xsl:call-template name="generate-task-content"/></task>
            </xsl:when>
            <xsl:when test="$infotype='reference'">
              <xsl:call-template name="use-reference-dtd"/>
              <reference><xsl:call-template name="add-language-attribute"/><xsl:call-template name="generate-reference-content"/></reference>
            </xsl:when>
            <xsl:otherwise>
              <xsl:call-template name="use-topic-dtd"/>
              <topic><xsl:call-template name="add-language-attribute"/><xsl:call-template name="generate-topic-content"/></topic>
            </xsl:otherwise>
          </xsl:choose>
        </saxon:output>
      </xsl:when>
      <xsl:otherwise>
        <xsl:choose>
          <xsl:when test="$infotype='concept'">
            <concept><xsl:call-template name="add-language-attribute"/><xsl:call-template name="generate-concept-content"/></concept>
          </xsl:when>
          <xsl:when test="$infotype='task'">
            <task><xsl:call-template name="add-language-attribute"/><xsl:call-template name="generate-task-content"/></task>
          </xsl:when>
          <xsl:when test="$infotype='reference'">
            <reference><xsl:call-template name="add-language-attribute"/><xsl:call-template name="generate-reference-content"/></reference>
          </xsl:when>
          <xsl:otherwise>
            <topic><xsl:call-template name="add-language-attribute"/><xsl:call-template name="generate-topic-content"/></topic>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:if>
</xsl:template>

<!-- ********************** [4] LISTS AND RELATED ELEMENTS ********************** -->

<xsl:template name="output-standard-ol">
  <xsl:apply-templates select="i1|i2|i3|iref" mode="ensure-proper-placement"/>
  <xsl:value-of select="$newline"/>
  <ol>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:call-template name="add-compact-attribute-for-list"/>
    <!-- OL will not have any children when @conloc is specified, so add <li/> -->
    <xsl:if test="@conloc"><li/></xsl:if>
    <xsl:apply-templates/>
    <xsl:value-of select="$newline"/>
  </ol>
  <xsl:apply-templates select="fn" mode="ensure-proper-placement"/>
</xsl:template>

<!-- Determine the proper wrapper, if any, and output a standard OL -->
<xsl:template match="ol">
  <xsl:param name="parent-element"/>
  <xsl:if test="@seq | @dbscalepct ">
    <!-- output-message: unsupported attributes -->
  </xsl:if>
  <xsl:if test="@oltype[not(contains(@oltype,'normal'))]">
    <!-- output-message, unless in task; then, should use steps/substeps -->
  </xsl:if>
  <xsl:choose>
    <xsl:when test="$parent-element='refbody'">
      <section><xsl:call-template name="output-standard-ol"/></section>
    </xsl:when>
    <xsl:when test="$parent-element='conbody' and preceding-sibling::pblk">
      <section><xsl:call-template name="output-standard-ol"/></section>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="output-standard-ol"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- This template will only match the first OL in a task, which should be mapped to a <steps> element. 
     Only the first OL is called with mode=output-as-steps, all others are placed in context or result. -->
<xsl:template match="ol" mode="output-as-steps">
  <steps>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:if test="@conloc">        <!-- OL will not have any children, so add required elements -->
      <step><cmd/></step>
    </xsl:if>
    <xsl:apply-templates mode="output-as-steps"/>
  </steps>
</xsl:template>

<xsl:template name="output-standard-ul">
  <xsl:variable name="el-name">
    <xsl:choose>
      <xsl:when test="contains(@ultype,'simple') and
                      not(li/address|li/annot|li/attention|li/bridge|li/caution|li/cgraphic|
                          li/danger|li/dl|li/fig|li/gl|li/lines|li/lq|li/mknote|li/modinfo|li/note|
                          li/notelist|li/ol|li/p|li/parml|li/pblk|li/screen|li/syntax|li/table|li/ul|li/xmp|
                          liblk/bridge|liblk/title)">
        <xsl:text>sl</xsl:text>
      </xsl:when>
      <xsl:when test="contains(@ultype,'simple')">
        <!-- output-message that we can't save simple type -->
        <xsl:text>ul</xsl:text>
      </xsl:when>
      <xsl:otherwise>ul</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:value-of select="$newline"/>
  <xsl:apply-templates select="i1|i2|i3|iref" mode="ensure-proper-placement"/>
  <xsl:element name="{$el-name}">
    <xsl:call-template name="add-default-attributes"/>
    <xsl:call-template name="add-compact-attribute-for-list"/>
    <xsl:if test="@conloc">        <!-- OL will not have any children, so add <li/> -->
      <xsl:if test="$el-name='ul'"><li/></xsl:if>
      <xsl:if test="$el-name='sl'"><sli/></xsl:if>
    </xsl:if>
    <xsl:apply-templates>
      <xsl:with-param name="ultype"><xsl:value-of select="$el-name"/></xsl:with-param>
    </xsl:apply-templates>
    <xsl:value-of select="$newline"/>
  </xsl:element>
  <xsl:apply-templates select="fn" mode="ensure-proper-placement"/>
</xsl:template>

<!-- Determine the proper wrapper, if any, and output a standard UL -->
<xsl:template match="ul">
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <xsl:when test="$parent-element='refbody'">
      <section><xsl:call-template name="output-standard-ul"/></section>
    </xsl:when>
    <xsl:when test="$parent-element='conbody' and preceding-sibling::pblk">
      <section><xsl:call-template name="output-standard-ul"/></section>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="output-standard-ul"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="output-standard-notelist">
  <xsl:value-of select="$newline"/>
  <note>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:if test="title and not(@conloc)">
      <xsl:attribute name="type">other</xsl:attribute>
      <xsl:attribute name="othertype"><xsl:value-of select="title"/></xsl:attribute>
    </xsl:if>
    <xsl:apply-templates select="i1|i2|i3|iref" mode="ensure-proper-placement"/>
    <xsl:if test="not(@conloc)">
      <ol>
        <xsl:call-template name="add-compact-attribute-for-list"/>
        <xsl:apply-templates select="*[not(name()='title')]"/>
      </ol>
      <xsl:apply-templates select="fn" mode="ensure-proper-placement"/>
    </xsl:if>
    <xsl:value-of select="$newline"/>
  </note>
</xsl:template>

<!-- Determine the proper wrapper, if any, and map notelist to <note><ol/></note> -->
<xsl:template match="notelist">
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <xsl:when test="$parent-element='refbody'">
      <section><xsl:call-template name="output-standard-notelist"/></section>
    </xsl:when>
    <xsl:when test="$parent-element='conbody' and preceding-sibling::pblk">
      <section><xsl:call-template name="output-standard-notelist"/></section>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="output-standard-notelist"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- LI maps directly to LI, unless it is in a list of steps (below) -->
<xsl:template match="li">
  <xsl:param name="ultype"/>
  <xsl:value-of select="$newline"/>
  <xsl:choose>
    <xsl:when test="$ultype='sl'">
      <sli>
        <xsl:call-template name="add-default-attributes"/>
        <xsl:apply-templates/>
      </sli>
    </xsl:when>
    <xsl:otherwise>
      <li>
        <xsl:call-template name="add-default-attributes"/>
        <xsl:apply-templates/>
      </li>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- If the LI contains only text or phrases, place them in the <cmd> element. If it contains
     block-like element, we cannot determine which part is the command; so, place it in
     <required-cleanup> in the info section. Users should move the command portion into
     the <cmd> element. -->
<xsl:template match="li" mode="output-as-steps">
  <xsl:value-of select="$newline"/>
  <step>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:choose>
      <xsl:when test="address|attention|bridge|caution|cgraphic|danger|dl|fig|
                      gl|lines|litdata|lq|mknote|modinfo|note|notelist|ol|p|parml|
                      pblk|screen|syntax|table|ul|xmp">
        <cmd/>
        <info>
          <xsl:call-template name="output-required-cleanup-warning"/>
          <required-cleanup>
            <xsl:apply-templates/>
          </required-cleanup>
        </info>
      </xsl:when>
      <xsl:otherwise>
        <cmd><xsl:apply-templates/></cmd>
      </xsl:otherwise>
    </xsl:choose>
  </step>
</xsl:template>

<!-- Move indices, titles, and bridges outside of the DL. Then map DL to DL, saving
     the default and compact attributes. -->
<xsl:template name="output-standard-dl">
  <!-- <xsl:apply-templates select="i1|i2|i3|iref|dlentry/i1|dlentry/i2|dlentry/i3|dlentry/iref" mode="ensure-proper-placement"/> -->
  <xsl:value-of select="$newline"/>
  <xsl:apply-templates select="i1|i2|i3|iref" mode="ensure-proper-placement"/>
  <dl>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:call-template name="add-compact-attribute-for-list"/>
    <xsl:if test="@conloc">    <!-- If using @conloc, ensure required elements are present -->
      <dlentry><dt/><dd/></dlentry>
    </xsl:if>
    <!-- IDDoc requires both headers or neither, so if one is present both are -->
    <xsl:if test="termhd | defnhd">
      <dlhead>
        <dthd>
          <xsl:apply-templates select="termhd"/>
        </dthd>
        <ddhd>
          <xsl:apply-templates select="defnhd"/>
        </ddhd>
      </dlhead>
    </xsl:if>
    <!-- Fallthrough processing for bridge/title/indices will not output them here -->
    <xsl:apply-templates select="*[not(name()='termhd' or name()='defnhd')]"/>
    <xsl:value-of select="$newline"/>
  </dl>
  <xsl:apply-templates select="fn|dlentry/fn" mode="ensure-proper-placement"/>
</xsl:template>

<!-- Determine the proper wrapper, if any, and output a standard DL -->
<xsl:template match="dl">
  <xsl:if test="@headstyle | @termstyle | termwidth">
    <!-- output-message, unless we can put phrases for @headstyle, @termstyle -->
  </xsl:if>
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <xsl:when test="$parent-element='refbody'">
      <section><xsl:call-template name="output-standard-dl"/></section>
    </xsl:when>
    <xsl:when test="$parent-element='conbody' and preceding-sibling::pblk">
      <section><xsl:call-template name="output-standard-dl"/></section>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="output-standard-dl"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="dl/termhd | dl/defnhd">
  <xsl:apply-templates/>
</xsl:template>

<xsl:template match="dlentry">
  <xsl:value-of select="$newline"/>
  <dlentry>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:if test="@conloc">
      <dt/><dd/>
    </xsl:if>
    <xsl:apply-templates/>
  </dlentry>
</xsl:template>

<xsl:template match="dlentry/term">
  <dt>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates/>
  </dt>
</xsl:template>

<xsl:template match="dlentry/defn">
  <xsl:value-of select="$newline"/>
  <dd>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:call-template name="pull-indices-to-defn"/>
    <xsl:apply-templates>
      <xsl:with-param name="parent-element">dd</xsl:with-param>
    </xsl:apply-templates>
  </dd>
</xsl:template>

<xsl:template name="output-standard-parml">
  <xsl:value-of select="$newline"/>
  <xsl:apply-templates select="i1|i2|i3|iref" mode="ensure-proper-placement"/>
  <parml>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:call-template name="add-compact-attribute-for-list"/>
    <xsl:if test="@conloc">
      <plentry><pt/><pd/></plentry>
    </xsl:if>
    <xsl:if test="termhd | defnhd">
      <plentry>
        <pt>
          <xsl:apply-templates select="termhd"/>
        </pt>
        <pd>
          <xsl:apply-templates select="defnhd"/>
        </pd>
      </plentry>
    </xsl:if>
    <xsl:apply-templates select="*[not(name()='termhd' or name()='defnhd')]"/>
    <xsl:value-of select="$newline"/>
  </parml>
  <xsl:apply-templates select="fn|parm/fn" mode="ensure-proper-placement"/>
</xsl:template>

<!-- Similar to DL: output wrapper, then map to PARML -->
<xsl:template match="parml">
  <xsl:param name="parent-element"/>
  <xsl:if test="@headstyle | @termstyle | termwidth">
    <!-- output-message, unless we can put phrases for @headstyle, @termstyle -->
  </xsl:if>
  <xsl:choose>
    <xsl:when test="$parent-element='refbody'">
      <section><xsl:call-template name="output-standard-parml"/></section>
    </xsl:when>
    <xsl:when test="$parent-element='conbody' and preceding-sibling::pblk">
      <section><xsl:call-template name="output-standard-parml"/></section>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="output-standard-parml"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="parm/termhd | parm/defnhd">
  <xsl:call-template name="add-default-attributes"/>
  <xsl:apply-templates/>
</xsl:template>

<xsl:template match="parm">
  <xsl:value-of select="$newline"/>
  <plentry>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:if test="@conloc">
      <pt/><pd/>
    </xsl:if>
    <xsl:apply-templates/>
  </plentry>
</xsl:template>

<xsl:template match="parm/term">
  <pt>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates/>
  </pt>
</xsl:template>

<xsl:template match="parm/defn">
  <xsl:value-of select="$newline"/>
  <pd>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:call-template name="pull-indices-to-defn"/>
    <xsl:apply-templates>
      <xsl:with-param name="parent-element">pd</xsl:with-param>
    </xsl:apply-templates>
  </pd>
</xsl:template>

<xsl:template name="output-standard-gl">
  <xsl:if test="@retkey">
    <!-- output-message -->
  </xsl:if>
  <xsl:value-of select="$newline"/>
  <xsl:apply-templates select="i1|i2|i3|iref" mode="ensure-proper-placement"/>
  <dl>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:call-template name="add-compact-attribute-for-list"/>
    <xsl:if test="@conloc">
      <dlentry><dt/><dd/></dlentry>
    </xsl:if>
    <xsl:apply-templates/>
    <xsl:value-of select="$newline"/>
  </dl>
  <xsl:apply-templates select="fn|glentry/fn" mode="ensure-proper-placement"/>
</xsl:template>

<!-- DITA does not yet have a good mapping for GL, so map it to DL -->
<xsl:template match="gl">
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <xsl:when test="contains(@spec,'auto')">
      <!-- output-message -->
    </xsl:when>
    <xsl:when test="$parent-element='refbody'">
      <section><xsl:call-template name="output-standard-gl"/></section>
    </xsl:when>
    <xsl:when test="$parent-element='conbody' and preceding-sibling::pblk">
      <section><xsl:call-template name="output-standard-gl"/></section>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="output-standard-gl"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="glentry">
  <xsl:value-of select="$newline"/>
  <dlentry>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:if test="@conloc">
      <dt/><dd/>
    </xsl:if>
    <xsl:apply-templates/>
  </dlentry>
</xsl:template>

<xsl:template match="glentry/term">
  <dt>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates/>
  </dt>
</xsl:template>

<xsl:template match="glentry/defn">
  <xsl:value-of select="$newline"/>
  <dd>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:call-template name="pull-indices-to-defn"/>
    <xsl:apply-templates/>
  </dd>
</xsl:template>

<xsl:template match="liblk | dlblk | parmblk | glblk">
  <!-- output-message about conloc? May have to just output children inline. -->
  <xsl:apply-templates/>
</xsl:template>

<xsl:template match="liblk" mode="output-as-steps">
  <xsl:apply-templates mode="output-as-steps"/>
</xsl:template>


<!-- Add @compact for lists. Can be taken from linespace on the list, or linespace on an oldef-type
     element in the document prolog. -->
<xsl:template name="add-compact-attribute-for-list">
  <xsl:variable name="linespacevalue">  <!-- Get the linespace value -->
    <xsl:choose>
      <xsl:when test="@linespace">
        <xsl:value-of select="@linespace"/>
      </xsl:when>
      <xsl:when test="(self::ol or self::notelist) and @def and /ibmiddoc/prolog/propdefs/oldef[@defname=current()/@def]/@linespace">
        <xsl:value-of select="/ibmiddoc/prolog/propdefs/oldef[@defname=current()/@def]/@linespace"/>
      </xsl:when>
      <xsl:when test="(self::ol or self::notelist) and /ibmiddoc/prolog/propdefs/oldef[not(@defname)]/@linespace">
        <xsl:value-of select="/ibmiddoc/prolog/propdefs/oldef[not(@defname)]/@linespace"/>
      </xsl:when>
      <xsl:when test="self::ul and @def and /ibmiddoc/prolog/propdefs/uldef[@defname=current()/@def]/@linespace">
        <xsl:value-of select="/ibmiddoc/prolog/propdefs/uldef[@defname=current()/@def]/@linespace"/>
      </xsl:when>
      <xsl:when test="self::ul and /ibmiddoc/prolog/propdefs/uldef[not(@defname)]/@linespace">
        <xsl:value-of select="/ibmiddoc/prolog/propdefs/uldef[not(@defname)]/@linespace"/>
      </xsl:when>
      <xsl:when test="(self::dl or self::parml) and @def and /ibmiddoc/prolog/propdefs/dldef[@defname=current()/@def]/@linespace">
        <xsl:value-of select="/ibmiddoc/prolog/propdefs/dldef[@defname=current()/@def]/@linespace"/>
      </xsl:when>
      <xsl:when test="(self::dl or self::parml) and /ibmiddoc/prolog/propdefs/dldef[not(@defname)]/@linespace">
        <xsl:value-of select="/ibmiddoc/prolog/propdefs/dldef[not(@defname)]/@linespace"/>
      </xsl:when>
      <xsl:when test="self::gl and @def and /ibmiddoc/prolog/propdefs/gldef[@defname=current()/@def]/@linespace">
        <xsl:value-of select="/ibmiddoc/prolog/propdefs/gldef[@defname=current()/@def]/@linespace"/>
      </xsl:when>
      <xsl:when test="self::gl and /ibmiddoc/prolog/propdefs/gldef[not(@defname)]/@linespace">
        <xsl:value-of select="/ibmiddoc/prolog/propdefs/gldef[not(@defname)]/@linespace"/>
      </xsl:when>
    </xsl:choose>
  </xsl:variable>
  <!-- If a linespace value exists, create the compact attribute in DITA -->
  <xsl:if test="string-length($linespacevalue) > 0">
    <xsl:attribute name="compact">
      <xsl:if test="contains($linespacevalue,'compact')">yes</xsl:if>
      <xsl:if test="contains($linespacevalue,'space')">no</xsl:if>
    </xsl:attribute>
  </xsl:if>
</xsl:template>

<!-- ************************** [5] TABLES ************************************ -->
<!-- Each table applies-templates on every attribute. Most should be copied through
     as-is to DITA, because they are CALS standards. Any that should not be copied
     do not have a matching template with mode="table-attribute".              

     Most table elements are also copied through as-is, with some small exceptions
     (such as mapping cap to title). -->
<xsl:template name="output-standard-table">
  <xsl:value-of select="$newline"/>
  <table>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:call-template name="add-scale-attribute"/>
    <xsl:apply-templates select="@pgwide"/>
    <!-- allow @frame to fallthrough as-is; values match in IDD and DITA -->
    <xsl:apply-templates select="@*" mode="table-attribute"/>
    <xsl:if test="@conloc">  <!-- Create minimum required tags/attributes if using conloc -->
      <tgroup cols="1"><tbody><row><entry/></row></tbody></tgroup>
    </xsl:if>
    <xsl:apply-templates/>
    <xsl:value-of select="$newline"/>
  </table>
</xsl:template>

<xsl:template match="table/@pgwide">
  <xsl:attribute name="expanse">
    <xsl:choose>
      <xsl:when test="contains(.,'0')">column</xsl:when>
      <xsl:when test="contains(.,'1')">page</xsl:when>
      <xsl:when test="contains(.,'2')">textline</xsl:when>
    </xsl:choose>
  </xsl:attribute>
</xsl:template>

<!-- Output a wrapper, if needed, and then call the standard-table template. -->
<xsl:template match="table">
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <xsl:when test="$parent-element='conbody' and preceding-sibling::pblk">
      <section>
        <xsl:apply-templates select="i1|i2|i3|iref|tgroup/i1|tgroup/i2|tgroup/i3|tgroup/iref|
                                     tgroup/tbody/i1|tgroup/tbody/i2|tgroup/tbody/i3|tgroup/tbody/iref|
                                     tgroup/thead/i1|tgroup/thead/i2|tgroup/thead/i3|tgroup/thead/iref|
                                     tgroup/tfoot/i1|tgroup/tfoot/i2|tgroup/tfoot/i3|tgroup/tfoot/iref" mode="ensure-proper-placement">
          <xsl:with-param name="parent-element">section</xsl:with-param>
        </xsl:apply-templates>
        <xsl:call-template name="output-standard-table"/>
      </section>
    </xsl:when>
    <xsl:when test="parent::fn">
      <xsl:call-template name="output-required-cleanup-warning"/>
      <ph><required-cleanup>
        <xsl:apply-templates select="i1|i2|i3|iref|tgroup/i1|tgroup/i2|tgroup/i3|tgroup/iref|
                                     tgroup/tbody/i1|tgroup/tbody/i2|tgroup/tbody/i3|tgroup/tbody/iref|
                                     tgroup/thead/i1|tgroup/thead/i2|tgroup/thead/i3|tgroup/thead/iref|
                                     tgroup/tfoot/i1|tgroup/tfoot/i2|tgroup/tfoot/i3|tgroup/tfoot/iref" mode="ensure-proper-placement">
          <xsl:with-param name="parent-element">required-cleanup</xsl:with-param>
        </xsl:apply-templates>
        <xsl:call-template name="output-standard-table"/>
      </required-cleanup></ph>
    </xsl:when>
    <xsl:otherwise>
      <xsl:apply-templates select="i1|i2|i3|iref|tgroup/i1|tgroup/i2|tgroup/i3|tgroup/iref|
                                   tgroup/tbody/i1|tgroup/tbody/i2|tgroup/tbody/i3|tgroup/tbody/iref|
                                   tgroup/thead/i1|tgroup/thead/i2|tgroup/thead/i3|tgroup/thead/iref|
                                   tgroup/tfoot/i1|tgroup/tfoot/i2|tgroup/tfoot/i3|tgroup/tfoot/iref" mode="ensure-proper-placement">
        <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
      </xsl:apply-templates>
      <xsl:call-template name="output-standard-table"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="table/cap">
  <xsl:value-of select="$newline"/>
  <title>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates/>
  </title>
</xsl:template>

<xsl:template match="table/desc">
  <xsl:value-of select="$newline"/>
  <desc>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates/>
  </desc>
</xsl:template>

<xsl:template match="tgroup">
  <xsl:value-of select="$newline"/>
  <tgroup>
    <xsl:apply-templates select="@*" mode="table-attribute"/>
    <xsl:apply-templates/>
  <xsl:value-of select="$newline"/>
  </tgroup>
</xsl:template>

<xsl:template match="colspec">
  <xsl:value-of select="$newline"/>
  <xsl:element name="colspec">
    <xsl:apply-templates select="@*" mode="table-attribute"/>
  </xsl:element>
</xsl:template>

<xsl:template match="spanspec">
  <xsl:value-of select="$newline"/>
  <xsl:element name="spanspec">
    <xsl:apply-templates select="@*" mode="table-attribute"/>
  </xsl:element>
</xsl:template>

<xsl:template match="thead">
  <xsl:value-of select="$newline"/>
  <thead>
    <xsl:apply-templates select="@*" mode="table-attribute"/>
    <xsl:apply-templates/>
    <xsl:value-of select="$newline"/>
  </thead>
</xsl:template>

<xsl:template match="tfoot">
  <xsl:value-of select="$newline"/>
  <tfoot>
    <xsl:apply-templates select="@*" mode="table-attribute"/>
    <xsl:apply-templates/>
    <xsl:value-of select="$newline"/>
  </tfoot>
</xsl:template>

<xsl:template match="tbody">
  <xsl:value-of select="$newline"/>
  <tbody>
    <xsl:apply-templates select="@*" mode="table-attribute"/>
    <xsl:apply-templates/>
    <xsl:value-of select="$newline"/>
  </tbody>
</xsl:template>

<xsl:template match="row">
  <xsl:value-of select="$newline"/><row>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates select="@*" mode="table-attribute"/>
    <xsl:apply-templates/>
  </row>
</xsl:template>

<xsl:template match="entry">
  <xsl:value-of select="$newline"/>
  <entry>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates select="@*" mode="table-attribute"/>
    <xsl:if test="not(preceding-sibling::entry) and (../i1 or ../i2 or ../i3 or ../iref)">
      <xsl:apply-templates select="../i1|../i2|../i3|../iref" mode="ensure-proper-placement">
        <xsl:with-param name="parent-element">entry</xsl:with-param>
      </xsl:apply-templates>
    </xsl:if>
    <xsl:apply-templates/>
  </entry>
</xsl:template>

<!-- Table attributes that should be copied through as-is: -->
<xsl:template match="@colnum | @colname | @namest | @nameend | @spanname |
                     @align | @valign | @morerows | @colsep | @rowsep |
                     @rowheader | @colwidth | @cols | @frame" mode="table-attribute">
  <xsl:attribute name="{name()}"><xsl:value-of select="."/></xsl:attribute>
</xsl:template>

<xsl:template match="@conloc" mode="table-attribute">
  <!-- output-message -->
</xsl:template>

<xsl:template match="@*" mode="table-attribute"/>

<!-- ************************** [6] FIGURES ************************************ -->

<xsl:template name="output-standard-fig">
  <xsl:param name="parent-element"/>
  <xsl:variable name="pgwide">
    <xsl:call-template name="get-pgwide-attribute"/>
  </xsl:variable>
  <xsl:value-of select="$newline"/>
  <!-- Indices need to be moved out of the figure -->
  <xsl:if test="i1|i2|i3|iref">
    <xsl:choose>
      <!-- If the figure is already in a section or example, output indices directely -->
      <xsl:when test="$parent-element='refbody' or ($parent-element='conbody' and preceding-sibling::pblk)">
        <xsl:apply-templates select="i1|i2|i3|iref" mode="ensure-proper-placement"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:call-template name="output-required-cleanup-warning"/>
        <required-cleanup><xsl:apply-templates select="i1|i2|i3|iref" mode="ensure-proper-placement"/></required-cleanup>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:if>
  <!-- Any foonotes before the title or desc must be moved out -->
  <xsl:apply-templates select="fn[following-sibling::cap]|fn[following-sibling::desc]" mode="ensure-proper-placement"/>
  <fig>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:call-template name="add-frame-attribute"/>
    <xsl:call-template name="add-scale-attribute"/>
    <xsl:if test="string-length($pgwide) > 0">
      <xsl:attribute name="expanse">
        <xsl:choose>
          <xsl:when test="contains($pgwide,'0')">column</xsl:when>
          <xsl:when test="contains($pgwide,'1')">page</xsl:when>
          <xsl:otherwise>page</xsl:otherwise> <!-- output-message for invalid value? -->
        </xsl:choose>
      </xsl:attribute>
    </xsl:if>
    <!-- Process fig contents. Indices and selected footnotes will be ignored. -->
    <xsl:apply-templates/>
  </fig>
</xsl:template>

<!-- Determine the proper wrapper for fig, and then map it to fig -->
<xsl:template match="fig">
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <xsl:when test="$parent-element='refbody'">
      <example>
        <xsl:call-template name="output-standard-fig">
          <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
        </xsl:call-template>
      </example>
    </xsl:when>
    <xsl:when test="$parent-element='conbody' and preceding-sibling::pblk">
      <section>
        <xsl:call-template name="output-standard-fig">
          <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
        </xsl:call-template>
      </section>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="output-standard-fig">
        <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
      </xsl:call-template>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- figseg does not have a dita equivalent, so output the contents. -->
<xsl:template match="figseg">
  <!-- output-message? -->
  <xsl:apply-templates/>
</xsl:template>

<xsl:template match="fig/cap">
  <xsl:value-of select="$newline"/>
  <title>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates/>
  </title>
  <!-- E002962: move all indices from fig outside of it, before fig -->
  <!-- <xsl:apply-templates select="preceding-sibling::fn|preceding-sibling::i1|preceding-sibling::i2|
                               preceding-sibling::i3|preceding-sibling::iref"
                       mode="ensure-proper-placement"/> -->
</xsl:template>

<xsl:template match="fig/desc">
  <desc>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates/>
  </desc>
</xsl:template>

<!-- ********************** [7] NOTE ELEMENTS ********************************** -->

<xsl:template name="output-standard-note">
  <xsl:value-of select="$newline"/>
  <note>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:if test="title">
      <xsl:attribute name="type">other</xsl:attribute>
      <xsl:attribute name="othertype"><xsl:value-of select="title"/></xsl:attribute>
    </xsl:if>
    <xsl:apply-templates select="*[not(name()='title')]"/>
  </note>
</xsl:template>

<!-- Determine if a wrapper is needed, and output a default note. -->
<xsl:template match="note">
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <xsl:when test="$parent-element='refbody'">
      <section><xsl:call-template name="output-standard-note"/></section>
    </xsl:when>
    <xsl:when test="$parent-element='conbody' and preceding-sibling::pblk">
      <section><xsl:call-template name="output-standard-note"/></section>
    </xsl:when>
    <xsl:when test="$parent-element='shortdesc'">
      <xsl:call-template name="output-required-cleanup-warning"/>
      <ph><required-cleanup><xsl:call-template name="output-standard-note"/></required-cleanup></ph>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="output-standard-note"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="notebody">
  <xsl:apply-templates>
    <xsl:with-param name="parent-element">note</xsl:with-param>
  </xsl:apply-templates>
</xsl:template>

<!-- Anything that will be mapped to NOTE with a recognized type can use this function. The
     actual template match creates a wrapper, if needed, and calls one of the 3 templates
     below to set the type; each of those calls this template with the type. -->
<xsl:template name="output-note-with-type">
  <xsl:param name="notetype">note</xsl:param>
  <xsl:value-of select="$newline"/>
  <note type="{$notetype}">
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates>
      <xsl:with-param name="parent-element">note</xsl:with-param>
    </xsl:apply-templates>
  </note>
</xsl:template>

<xsl:template name="output-standard-attention">
  <xsl:call-template name="output-note-with-type">
    <xsl:with-param name="notetype">attention</xsl:with-param>
  </xsl:call-template>
</xsl:template>
<xsl:template name="output-standard-caution">
  <xsl:call-template name="output-note-with-type">
    <xsl:with-param name="notetype">caution</xsl:with-param>
  </xsl:call-template>
</xsl:template>
<xsl:template name="output-standard-danger">
  <xsl:call-template name="output-note-with-type">
    <xsl:with-param name="notetype">danger</xsl:with-param>
  </xsl:call-template>
</xsl:template>

<xsl:template match="attention | warning">
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <xsl:when test="$parent-element='refbody'">
      <section><xsl:call-template name="output-standard-attention"/></section>
    </xsl:when>
    <xsl:when test="$parent-element='conbody' and preceding-sibling::pblk">
      <section><xsl:call-template name="output-standard-attention"/></section>
    </xsl:when>
    <xsl:when test="$parent-element='shortdesc'">
      <xsl:call-template name="output-required-cleanup-warning"/>
      <ph><required-cleanup><xsl:call-template name="output-standard-attention"/></required-cleanup></ph>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="output-standard-attention"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="caution">
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <xsl:when test="$parent-element='refbody'">
      <section><xsl:call-template name="output-standard-caution"/></section>
    </xsl:when>
    <xsl:when test="$parent-element='conbody' and preceding-sibling::pblk">
      <section><xsl:call-template name="output-standard-caution"/></section>
    </xsl:when>
    <xsl:when test="$parent-element='shortdesc'">
      <xsl:call-template name="output-required-cleanup-warning"/>
      <ph><required-cleanup><xsl:call-template name="output-standard-caution"/></required-cleanup></ph>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="output-standard-caution"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="danger">
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <xsl:when test="$parent-element='refbody'">
      <section><xsl:call-template name="output-standard-danger"/></section>
    </xsl:when>
    <xsl:when test="$parent-element='conbody' and preceding-sibling::pblk">
      <section><xsl:call-template name="output-standard-danger"/></section>
    </xsl:when>
    <xsl:when test="$parent-element='shortdesc'">
      <xsl:call-template name="output-required-cleanup-warning"/>
      <ph><required-cleanup><xsl:call-template name="output-standard-danger"/></required-cleanup></ph>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="output-standard-danger"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- ***************************** [8] TITLES ****************************** -->
<!-- Titles can appear in a wide variety of locations in IDDoc, so this is a holder
     for titles that do not easily fall into a category. Titles that are processed
     with their parents are listed here, with the proper section:
     
     /ibmiddoc/prolog/ibmbibentry/doctitle/titleblk/title before [1]
     d/dprolog/titleblk/title in [3]
     fragref/title in [11]
     group/title in [11]
     proc/titleblk/title [13]
     partasm/title [15]
     moditemdef/title [18]
     title (mode="resolve-cit"): all titles within <cit> tags [20]
     subtitle (mode="resolve-cit"): all subtitles within <cit> tags [20]
     -->

<!-- Handle the title of frontmatter div-like elements, such as Preface. -->
<xsl:template match="frontm/*/specdprolog/titleblk/title">
  <xsl:call-template name="output-title-as-title"/>
</xsl:template>

<!-- Outside of frontmatter, abstract is mapping to shortdesc, so it must be ph-like -->
<xsl:template match="*[not(name()='frontm')]/abstract/specdprolog/titleblk/title">
  <ph><b>
    <xsl:apply-templates/>
  </b></ph>
</xsl:template>

<xsl:template match="ednotices/title">
  <xsl:value-of select="$newline"/>
  <title>
    <xsl:apply-templates/>
  </title>
</xsl:template>

<xsl:template match="safety/titleblk/title">
  <xsl:call-template name="output-title-as-title"/>
</xsl:template>

<!-- Titles in list blocks cannot be converted directly to DITA. 
     E003814: Move the titles into a container appropriate for this list.
              Continue to put it in required-cleanup. -->
<xsl:template match="liblk/title">
  <xsl:call-template name="output-required-cleanup-warning"/>
  <li><required-cleanup><xsl:apply-templates/></required-cleanup></li>
</xsl:template>
<xsl:template match="liblk/title" mode="output-as-steps">
  <xsl:call-template name="output-required-cleanup-warning"/>
  <step><cmd><required-cleanup><xsl:apply-templates/></required-cleanup></cmd></step>
</xsl:template>
<xsl:template match="dlblk/title|glblk/title" priority="1">
  <xsl:call-template name="output-required-cleanup-warning"/>
  <dlentry><dt/><dd><required-cleanup><xsl:apply-templates/></required-cleanup></dd></dlentry>
</xsl:template>
<xsl:template match="parmblk/title" priority="1">
  <xsl:call-template name="output-required-cleanup-warning"/>
  <plentry><pt/><pd><required-cleanup><xsl:apply-templates/></required-cleanup></pd></plentry>
</xsl:template>

<xsl:template match="doctitle">
  <xsl:apply-templates/>
</xsl:template>

<xsl:template match="library">
  <xsl:apply-templates/>
</xsl:template>

<xsl:template match="titleblk">
  <xsl:apply-templates/>
</xsl:template>

<xsl:template match="title">
  <xsl:call-template name="output-title-as-title"/>
</xsl:template>

<!-- The first pass should resolve any gendtitle elements to the proper text -->
<xsl:template match="gendtitle">
  <title>&nbsp;</title>
</xsl:template>

<!-- ********************** [9] NON-WRAPPING ELEMENTS ********************** -->

<xsl:template name="output-standard-xmp">
  <xsl:variable name="pgwide">
    <xsl:call-template name="get-pgwide-attribute"/>
  </xsl:variable>
  <xsl:value-of select="$newline"/>
  <codeblock>
    <xsl:call-template name="add-default-attributes"/>
    <!-- output-message for linelength, notation: unsupported attributes -->
    <xsl:if test="string-length($pgwide)>0">
      <xsl:attribute name="expanse">
        <xsl:choose>
          <xsl:when test="contains($pgwide,'1')">page</xsl:when>
          <xsl:when test="contains($pgwide,'2')">column</xsl:when>
          <xsl:otherwise>column</xsl:otherwise> <!-- output-message for invalid value? -->
        </xsl:choose>
      </xsl:attribute>
    </xsl:if>
    <xsl:apply-templates>
      <xsl:with-param name="parent-element">codeblock</xsl:with-param>
    </xsl:apply-templates>
  </codeblock>
</xsl:template>

<!-- Create wrapper if needed, and map xmp to codeblock -->
<xsl:template match="xmp">
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <xsl:when test="$parent-element='refbody'">
      <example><xsl:call-template name="output-standard-xmp"/></example>
    </xsl:when>
    <xsl:when test="$parent-element='conbody' and preceding-sibling::pblk">
      <section><xsl:call-template name="output-standard-xmp"/></section>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="output-standard-xmp"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Create wrapper and map to pre, without a new function. If more contexts are found
     that require wrappers, this may be moved to an output-standard-cgraphic template. -->
<xsl:template match="cgraphic">
  <xsl:param name="parent-element"/>
  <!-- output-message for @linelength, @notation? -->
  <xsl:value-of select="$newline"/>
  <xsl:choose>
    <xsl:when test="$parent-element='refbody' or
                    ($parent-element='conbody' and preceding-sibling::pblk)">
      <section><pre>
        <xsl:call-template name="add-default-attributes"/>
        <xsl:apply-templates>
          <xsl:with-param name="parent-element">pre</xsl:with-param>
        </xsl:apply-templates>
      </pre></section>
    </xsl:when>
    <xsl:otherwise>
      <pre>
        <xsl:call-template name="add-default-attributes"/>
        <xsl:apply-templates>
          <xsl:with-param name="parent-element">pre</xsl:with-param>
        </xsl:apply-templates>
      </pre>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="output-standard-screen">
  <xsl:variable name="pgwide">
    <xsl:call-template name="get-pgwide-attribute"/>
  </xsl:variable>
  <xsl:value-of select="$newline"/>
  <screen>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:if test="string-length($pgwide)>0">
      <xsl:attribute name="expanse">
        <xsl:choose>
          <xsl:when test="contains($pgwide,'1')">page</xsl:when>
          <xsl:when test="contains($pgwide,'2')">column</xsl:when>
          <xsl:otherwise>column</xsl:otherwise> <!-- output-message for invalid value? -->
        </xsl:choose>
      </xsl:attribute>
    </xsl:if>
    <xsl:apply-templates>
      <xsl:with-param name="parent-element">screen</xsl:with-param>
    </xsl:apply-templates>
  </screen>
</xsl:template>

<!-- Create any necessary wrapper and map screen to screen. -->
<xsl:template match="screen">
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <xsl:when test="$parent-element='refbody'">
      <section><xsl:call-template name="output-standard-screen"/></section>
    </xsl:when>
    <xsl:when test="$parent-element='conbody' and preceding-sibling::pblk">
      <section><xsl:call-template name="output-standard-screen"/></section>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="output-standard-screen"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Create any necessary wrapper and map lines to lines. -->
<xsl:template match="lines">
  <xsl:param name="parent-element"/>
  <!-- output-message for @notation? -->
  <xsl:value-of select="$newline"/>
  <xsl:choose>
    <xsl:when test="$parent-element='refbody' or
                    ($parent-element='conbody' and preceding-sibling::pblk)">
      <section><lines>
        <xsl:call-template name="add-default-attributes"/>
        <xsl:apply-templates/>
      </lines></section>
    </xsl:when>
    <xsl:otherwise>  
      <lines>
        <xsl:call-template name="add-default-attributes"/>
        <xsl:apply-templates/>
      </lines>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Litdata is mapping to pre, with any needed wrapper. This is a best-possible
     match that does not exactly fit, but litdata is not used too often. -->
<xsl:template match="litdata">
  <xsl:param name="parent-element"/>
  <!-- output-message for @notation? -->
  <xsl:choose>
    <xsl:when test="$parent-element='refbody'">
      <section><pre><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></pre></section>
    </xsl:when>
    <xsl:when test="$parent-element='conbody' and preceding-sibling::pblk">
      <section><pre><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></pre></section>
    </xsl:when>
    <xsl:when test="$parent-element='pre' or $parent-element='codeblock' or $parent-element='screen'">
      <ph><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></ph>
    </xsl:when>
    <xsl:otherwise>  
      <pre><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></pre>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- ****************************** [10] LERS ******************************-->

<!-- The outer LERS tag does not map to dita. Each LE child will create a reference topic. -->
<xsl:template match="lers">
  <xsl:apply-templates/>
</xsl:template>

<!-- Map LE to a reference topic. This can be overriden by placing a class on the LE element. -->
<xsl:template match="le">
  <xsl:call-template name="output-divlike">
    <xsl:with-param name="default-type">reference</xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!-- Treat LEN as a title; the subtitle portion of the function has no effect. -->
<xsl:template match="len">
  <xsl:call-template name="output-title-as-title"/>
</xsl:template>

<xsl:template match="ledesc">
  <xsl:value-of select="$newline"/>
  <section>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates>
      <xsl:with-param name="parent-element">section</xsl:with-param>
    </xsl:apply-templates>
  </section>
</xsl:template>

<!-- Map LEDI to section, unless it can be mapped to refsyn or example. 
     The id2xid pre-processor has added a title attribute to each LEDI element; this
     gives the generated (or user-specified) text for the ledi element. If for some reason
     the title is not specified, search for the title on the LERS ancestor; if not there,
     default to the class name. -->
<xsl:template match="ledi">
  <!-- output-message about @layout: unsupported attribute? -->
  <xsl:variable name="sectionname">
    <xsl:choose>
      <xsl:when test="@class='syntax' or @class='Syntax' or
                      @class='format' or @class='Format'">refsyn</xsl:when>
      <xsl:when test="@class='examples' or @class='Examples'">example</xsl:when>
      <xsl:otherwise>section</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:variable name="ledititle">  <!-- Title is placed on ledi's attribute by pre-processor -->
    <xsl:choose>
      <xsl:when test="@title"><xsl:value-of select="@title"/></xsl:when>
      <xsl:when test="@class='auth'"><xsl:value-of select="../../@auth"/></xsl:when>
      <xsl:when test="@class='comments'"><xsl:value-of select="../../@comments"/></xsl:when>
      <xsl:when test="@class='context'"><xsl:value-of select="../../@context"/></xsl:when>
      <xsl:when test="@class='defaults'"><xsl:value-of select="../../@defaults"/></xsl:when>
      <xsl:when test="@class='errcond'"><xsl:value-of select="../../@errcond"/></xsl:when>
      <xsl:when test="@class='examples'"><xsl:value-of select="../../@examples"/></xsl:when>
      <xsl:when test="@class='flags'"><xsl:value-of select="../../@flags"/></xsl:when>
      <xsl:when test="@class='format'"><xsl:value-of select="../../@format"/></xsl:when>
      <xsl:when test="@class='intrep'"><xsl:value-of select="../../@intrep"/></xsl:when>
      <xsl:when test="@class='messages'"><xsl:value-of select="../../@messages"/></xsl:when>
      <xsl:when test="@class='other'"><xsl:value-of select="../../@other"/></xsl:when>
      <xsl:when test="@class='parms'"><xsl:value-of select="../../@parms"/></xsl:when>
      <xsl:when test="@class='process'"><xsl:value-of select="../../@process"/></xsl:when>
      <xsl:when test="@class='purpose'"><xsl:value-of select="../../@purpose"/></xsl:when>
      <xsl:when test="@class='restrict'"><xsl:value-of select="../../@restrict"/></xsl:when>
      <xsl:when test="@class='results'"><xsl:value-of select="../../@retcodes"/></xsl:when>
      <xsl:when test="@class='sysenv'"><xsl:value-of select="../../@sysenv"/></xsl:when>
      <xsl:when test="@class='usage'"><xsl:value-of select="../../@usage"/></xsl:when>
      <xsl:when test="@class='version'"><xsl:value-of select="../../@version"/></xsl:when>
      <xsl:when test="not(parent::le/parent::lers)">
        <!-- output-message? should not get here (first condition should always fire) -->
        <xsl:value-of select="@class"/>
      </xsl:when>
    </xsl:choose>
  </xsl:variable>
  <xsl:value-of select="$newline"/>
  <xsl:element name="{$sectionname}">
    <xsl:call-template name="add-default-attributes"/>
    <title>
      <xsl:value-of select="$ledititle"/>
    </title>
    <xsl:apply-templates>
      <xsl:with-param name="parent-element">section</xsl:with-param>
    </xsl:apply-templates>
    <xsl:value-of select="$newline"/>
  </xsl:element>
</xsl:template>

<!-- ****************************** [11] SYNTAX *****************************-->

<!-- Most syntax elements have direct equivalents in DITA, although some massaging is necessary.
     For <syntax>, need to modify the pgwide and synstyle values to work with DITA. -->
<xsl:template name="output-standard-syntax">
  <xsl:variable name="pgwide">
    <xsl:call-template name="get-pgwide-attribute"/>
  </xsl:variable>
  <xsl:variable name="synstyle">
    <xsl:choose>
      <xsl:when test="@synstyle"><xsl:value-of select="@synstyle"/></xsl:when>
      <xsl:when test="@def and ancestor::*/*[contains(name(),'prolog')]/propdefs/figdef[@defname=current()/@def]/@synstyle">
        <xsl:value-of select="ancestor::*/*[contains(name(),'prolog')]/propdefs/figdef[@defname=current()/@def]/@synstyle"/>
      </xsl:when>
      <xsl:when test="ancestor::*/*[contains(name(),'prolog')]/propdefs/figdef[not(@defname)]/@synstyle">
        <xsl:value-of select="(ancestor::*/*[contains(name(),'prolog')]/propdefs/figdef[not(@defname)]/@synstyle)[last()]"/>
      </xsl:when>
    </xsl:choose>
  </xsl:variable>
  <xsl:value-of select="$newline"/>
  <syntaxdiagram>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:call-template name="add-scale-attribute"/>
    <!-- If a pgwide value is used, convert to the DITA expanse attribute -->
    <xsl:if test="string-length($pgwide) > 0">
      <xsl:attribute name="expanse">
        <xsl:choose>
          <xsl:when test="contains($pgwide,'1')">page</xsl:when>
          <xsl:when test="contains($pgwide,'2')">column</xsl:when>
          <xsl:otherwise>column</xsl:otherwise> <!-- output-message for invalid value? -->
        </xsl:choose>
      </xsl:attribute>
    </xsl:if>
    <!-- If synstyle is used, convert to the DITA frame attribute. Should call
         output-message that lblbox is not completely supported. -->
    <xsl:if test="string-length($synstyle)>0">
      <xsl:attribute name="frame">
        <xsl:choose>              <!-- box, lblbox, rule, space -->
          <xsl:when test="$synstyle='box'">all</xsl:when>
          <xsl:when test="$synstyle='lblbox'">all</xsl:when>
          <xsl:when test="$synstyle='rule'">topbot</xsl:when>
          <xsl:otherwise>none</xsl:otherwise>  <!-- space is default -->
        </xsl:choose>
      </xsl:attribute>
    </xsl:if>
    <xsl:apply-templates/>
  </syntaxdiagram>
</xsl:template>

<!-- Output a wrapper for syntax, and then call the default syntax processing. -->
<xsl:template match="syntax">
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <xsl:when test="$parent-element='refbody'">
      <refsyn>
        <xsl:apply-templates select="descendant::i1|descendant::i2|
                                     descendant::i3|descendant::iref" mode="ensure-proper-placement">
          <xsl:with-param name="parent-element">refsyn</xsl:with-param>
        </xsl:apply-templates>
        <xsl:call-template name="output-standard-syntax"/>
      </refsyn>
    </xsl:when>
    <xsl:when test="$parent-element='conbody' and preceding-sibling::pblk">
      <section>
        <xsl:apply-templates select="descendant::i1|descendant::i2|
                                     descendant::i3|descendant::iref" mode="ensure-proper-placement">
          <xsl:with-param name="parent-element">section</xsl:with-param>
        </xsl:apply-templates>
        <xsl:call-template name="output-standard-syntax"/>
      </section>
    </xsl:when>
    <xsl:when test="parent::fig">
      <p>
        <xsl:apply-templates select="descendant::i1|descendant::i2|
                                     descendant::i3|descendant::iref" mode="ensure-proper-placement">
          <xsl:with-param name="parent-element">p</xsl:with-param>
        </xsl:apply-templates>
        <xsl:call-template name="output-standard-syntax"/>
      </p>
    </xsl:when>
    <xsl:otherwise>
      <xsl:apply-templates select="descendant::i1|descendant::i2|
                                   descendant::i3|descendant::iref" mode="ensure-proper-placement">
        <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
      </xsl:apply-templates>
      <xsl:call-template name="output-standard-syntax"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="synblk">
  <synblk>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates/>
  </synblk>
</xsl:template>

<!-- The group element must map to groupchoice, groupcomp, or groupseq, based on the choiceseq
     attribute. The default mapping is to groupseq. @optreq is mapped to @importance. 
     Note: IDDoc references <repsep> tags, but DITA references them inline, so they must be
           copied into the group tag. -->
<xsl:template match="group">
  <xsl:variable name="groupel">
    <xsl:choose>
      <xsl:when test="@choiceseq='choice'">groupchoice</xsl:when>
      <xsl:when test="@choiceseq='composite'">groupcomp</xsl:when>
      <xsl:otherwise>groupseq</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:value-of select="$newline"/>
  <xsl:element name="{$groupel}">
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates select="@optreq"/>
    <!-- Process any title before the repsep tag is created -->
    <xsl:apply-templates select="title" mode="ensure-proper-placement"/>
    <xsl:if test="@repid and (ancestor::syntax/repsep[@id=current()/@repid] or
                              ancestor::synblk/repsep[@id=current()/@repid])">
      <!-- If @repid is specified and references a <repsep> within this diagram, copy that repsep here. 
           There should only be one, but the for-each means I don't have to keep re-selecting it. -->
      <xsl:for-each select="ancestor::*[name()='syntax' or name()='synblk']/repsep[@id=current()/@repid]">
        <repsep>
          <xsl:apply-templates select="@optreq"/>
          <xsl:value-of select="."/>
        </repsep>
      </xsl:for-each>
    </xsl:if>
    <xsl:apply-templates/>
  </xsl:element>
</xsl:template>

<!-- The group title must be processed first, before <repsep> tags are created. It is
     processed with mode=ensure-proper-placement, and it is ignored for fallthrough
     processing. -->
<xsl:template match="group/title"/>
<xsl:template match="group/title" mode="ensure-proper-placement">
  <title>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates/>
  </title>
</xsl:template>

<xsl:template match="fragment">
  <xsl:value-of select="$newline"/>
  <fragment>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates/>
  </fragment>
</xsl:template>

<!-- If @fragid is specified, convert it to an @href tag. Assumption is that the referenced
     fragment is within the same diagram, and thus within the same topic. -->
<xsl:template match="fragref[@fragid]">
  <xsl:variable name="current-topic">
    <xsl:call-template name="get-current-topic-id"/>
  </xsl:variable>
  <fragref href="#{$current-topic}/{@fragid}">
    <xsl:call-template name="add-default-attributes"/>
  </fragref>
</xsl:template>

<xsl:template match="fragref">   <!-- any fragref without @refid should have a <title> -->
  <fragref>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates/>
  </fragref>
</xsl:template>

<xsl:template match="fragref/title">  <!-- DITA fragref's contain text, not <title> -->
  <xsl:apply-templates/>
</xsl:template>

<!-- Copy these elements as-is, while converting @optreq to @importance -->
<xsl:template match="sep | var | delim | kwd | oper">
  <xsl:element name="{name()}">
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates select="@optreq"/>
    <xsl:apply-templates/>
  </xsl:element>
</xsl:template>

<xsl:template match="repsep"/>   <!-- pulled in from group -->

<xsl:template match="synnote">
  <xsl:value-of select="$newline"/>
  <synnote>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates/>
  </synnote>
</xsl:template>

<!-- Assumption: synnote can only reference a synnote within the same diagram -->
<xsl:template match="synnote[@refid]|xref[@objtype='synnote'][ancestor::syntax]">
  <xsl:variable name="current-topic">
    <xsl:call-template name="get-current-topic-id"/>
  </xsl:variable>
  <synnoteref href="#{$current-topic}/{@refid}"/>
</xsl:template>

<!-- Index terms and footnotes must be moved out of synph -->
<xsl:template match="synph">
  <xsl:apply-templates select="i1|i2|i3|iref|delim/i1|delim/i2|delim/i3|delim/iref|
                               kwd/i1|kwd/i2|kwd/i3|kwd/iref|oper/i1|oper/i2|oper/i3|oper/iref|
                               sep/i1|sep/i2|sep/i3|sep/iref|var/i1|var/i2|var/i3|var/iref"
                       mode="ensure-proper-placement"/>
  <synph>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates/>
  </synph>
  <xsl:apply-templates select="fn|sep/fn|var/fn|delim/fn|kwd/fn|oper/fn" mode="ensure-proper-placement"/>
</xsl:template>

<!-- ******************** [12] LINK/XREF ELEMENTS ********************* -->

<!-- Set up a list of all elements with IDs, for quick lookup when creating an xref, conref, etc -->
<xsl:key name="xreflist" match="*[@id]" use="@id"/>

<!-- Find out what element a reference is pointing to, and determine what the reference will
     be to (the name of the target element) in the DITA output file. If the target is a topic, determine
     what type of topic. -->
<xsl:template name="determine-link-target">
  <xsl:choose>
    <xsl:when test="@objtype='li' or @objtype='fig' or @objtype='table' or @objtype='fn'">
      <xsl:value-of select="@objtype"/>
    </xsl:when>
    <xsl:when test="@objtype='pblk' or @objtype='ledesc' or @objtype='msgitem' or @objtype='moditem'">
      <xsl:text>section</xsl:text>
    </xsl:when>
    <xsl:when test="key('xreflist',@refid)[1]/@chunk-filename">
      <xsl:for-each select="key('xreflist',@refid)[1]">
        <xsl:call-template name="determine-topic-type">
          <xsl:with-param name="default-type">topic</xsl:with-param>
        </xsl:call-template>
      </xsl:for-each>
    </xsl:when>
    <xsl:when test="@objtype='notloc'">url</xsl:when>
  </xsl:choose>
</xsl:template>

<!-- Elements are processed here when they are referenced. It will output the href
     value that should be used to reference this element. If the reference comes from
     the same chunk, the filename is not needed; otherwise, make it part of the href. -->
<xsl:template match="*" mode="get-xref-target">
  <xsl:param name="xreffile"/>  <!-- Pass in the file of the reference-ing element -->
  <xsl:variable name="thisfile"><xsl:call-template name="get-current-chunk-name"/></xsl:variable>
  <xsl:variable name="thistopic"><xsl:call-template name="get-current-topic-id"/></xsl:variable>
  <xsl:choose>
    
    <!-- If referencing a notloc: the contents of the notloc are the URL needed for the href.  -->
    <xsl:when test="self::notloc"><xsl:apply-templates/></xsl:when>

    <xsl:otherwise>
      <!-- If the source and target file are not the same, output the filename -->
      <xsl:if test="$thisfile!=$xreffile">
        <xsl:value-of select="$thisfile"/>
      </xsl:if>
      <xsl:text>#</xsl:text>
      <xsl:value-of select="$thistopic"/>   <!-- output the topic ID -->
      <xsl:if test="@id!=$thistopic">       <!-- If the target is not the topic itself, -->
        <xsl:text>/</xsl:text>              <!--        output the element ID as well.  -->
        <xsl:value-of select="@id"/>
      </xsl:if>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Get the target element (the DITA element), and set the type attribute. XREFs to a
     notloc (to the web) will not work properly, but this is not supported in any other
     IDDoc transform, so it should not appear in documents. 
     NOTE: xref to a synnote has @objtype=synnote, and is handled with ordinary 
           synnote processing.
     P016846: Move template content to "process-xref" to allow for required-cleanup. -->
<xsl:template match="xref">
  <xsl:choose>
    <!-- Cross-book links: First test is for /publish=no, other for /publish -->
    <xsl:when test="(key('xreflist',@refid))[1]/self::nameloc or contains(@tgtids,' DOCID=')">
      <xsl:call-template name="output-required-cleanup-warning"/>
      <required-cleanup><xsl:call-template name="process-xref"/></required-cleanup>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="process-xref"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- E003903: remove @type from the xref. It should be picked up during transforms. -->
<xsl:template name="process-xref">
  <xsl:variable name="target"><xsl:call-template name="determine-link-target"/></xsl:variable>
  <xsl:variable name="thisfile"><xsl:call-template name="get-current-chunk-name"/></xsl:variable>
  <xref>
    <xsl:call-template name="add-default-attributes"/>
    <!--<xsl:if test="string-length($target)>0 and $target!='url'">
      <xsl:attribute name="type"><xsl:value-of select="$target"/></xsl:attribute>
    </xsl:if>-->
    <xsl:if test="$target='url'">
      <xsl:attribute name="format">html</xsl:attribute>
    </xsl:if>
    <xsl:attribute name="href">
      <xsl:choose>
        <!-- If this is a cross-book link, using /PUBLISH -->
        <xsl:when test="contains(@tgtids,' DOCID=')">
          <xsl:text>unresolved-cross-book-link/</xsl:text>
          <!-- first get the target book's ID -->
          <xsl:value-of select="substring-before(substring-after(@tgtids,' DOCID='),' ')"/>
          <xsl:text>-</xsl:text>
          <!-- now get the target ID from the other book -->
          <xsl:value-of select="substring-before(substring-after(@tgtids,' ID='),' ')"/>
        </xsl:when>
        <!-- If this a cross-book link, NOT using /PUBLISH -->
        <xsl:when test="(key('xreflist',@refid))[1]/self::nameloc">
          <xsl:text>unresolved-cross-book-link/</xsl:text>
          <xsl:value-of select="(key('xreflist',@refid))[1]/nmlist/@docname"/>
          <xsl:text>-</xsl:text>
          <xsl:value-of select="(key('xreflist',@refid))[1]/nmlist"/>
        </xsl:when>
        <xsl:otherwise>  <!-- In case of duplicate IDs, link to the first -->
          <xsl:apply-templates select="(key('xreflist',@refid))[1]" mode="get-xref-target">
            <xsl:with-param name="xreffile"><xsl:value-of select="$thisfile"/></xsl:with-param>
          </xsl:apply-templates>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:attribute>
    <!-- E003374: use xreftext if it was specified on the target -->
    <xsl:if test="(key('xreflist',@refid))[1]/@xreftext">
      <xsl:value-of select="(key('xreflist',@refid))[1]/@xreftext"/>
    </xsl:if>
  </xref>
</xsl:template>

<!-- Standard processing for <l> is to map to xref: all links within the document use XREF in
     DITA. The scope attribute may be set by @class (process described below). @format is set
     to URL for any links created with NOTLOC; otherwise it is left blank. @type is determined
     as with the XREF element. 
     E003903: remove @type from xref. It should be picked up during transforms. -->
<xsl:template name="output-l-as-xref">
  <xsl:variable name="target"><xsl:call-template name="determine-link-target"/></xsl:variable>
  <xsl:variable name="thisfile"><xsl:call-template name="get-current-chunk-name"/></xsl:variable>
  <!-- To get the scope attribute: First check this tag for @class. If unspecified, check to
       see if a default is specified on the LDESCS tag which contains the referenced notloc. If
       not, default to external for url's, and default to empty (unspecified) for anything else. -->
  <xsl:variable name="scope">
    <xsl:choose>
      <xsl:when test="@class='newwindow'">external</xsl:when>
      <xsl:when test="@class='samewindow' or @class='fullwindow'">local</xsl:when>
      <xsl:when test="$target='url' and key('xreflist',@linkend)/../@class='newwindow'">external</xsl:when>
      <xsl:when test="$target='url' and key('xreflist',@linkend)/../@class='samewindow'">local</xsl:when>
      <xsl:when test="$target='url' and key('xreflist',@linkend)/../@class='fullwindow'">local</xsl:when>
      <xsl:when test="$target='url'">local</xsl:when>
    </xsl:choose>
  </xsl:variable>
  <xref>
    <xsl:call-template name="add-default-attributes"/>
    <!--<xsl:if test="string-length($target)>0 and $target!='url'">
      <xsl:attribute name="type"><xsl:value-of select="$target"/></xsl:attribute>
    </xsl:if>-->
    <xsl:if test="string-length($scope)>0">
      <xsl:attribute name="scope"><xsl:value-of select="$scope"/></xsl:attribute>
    </xsl:if>
    <xsl:if test="$target='url'">
      <xsl:attribute name="format">html</xsl:attribute>
    </xsl:if>
    <xsl:attribute name="href">
      <xsl:choose>
        
        <!-- If this is a cross-book link, using /PUBLISH -->
        <xsl:when test="contains(@tgtids,' DOCID=')">
          <xsl:text>unresolved-cross-book-link/</xsl:text>
          <!-- first get the target book's ID -->
          <xsl:value-of select="substring-before(substring-after(@tgtids,' DOCID='),' ')"/>
          <xsl:text>-</xsl:text>
          <!-- now get the target ID from the other book -->
          <xsl:value-of select="substring-before(substring-after(@tgtids,' ID='),' ')"/>
        </xsl:when>
        <!-- If this a cross-book link, NOT using /PUBLISH -->
        <xsl:when test="(key('xreflist',@linkend))[1]/self::nameloc">
          <xsl:text>unresolved-cross-book-link/</xsl:text>
          <xsl:value-of select="(key('xreflist',@linkend))[1]/nmlist/@docname"/>
          <xsl:text>-</xsl:text>
          <xsl:value-of select="(key('xreflist',@linkend))[1]/nmlist"/>
        </xsl:when>
        <xsl:otherwise>
          <!-- In case of duplicate IDs, link to the first -->
          <xsl:apply-templates select="key('xreflist',@linkend)[1]" mode="get-xref-target">
            <xsl:with-param name="xreffile"><xsl:value-of select="$thisfile"/></xsl:with-param>
          </xsl:apply-templates>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:attribute>
    <xsl:apply-templates/>
  </xref>
</xsl:template>

<!-- Create any needed wrapper, and map the <l> to an xref.
     P016846: Move template content to "process-xref" to allow for required-cleanup. -->
<xsl:template match="l">
  <xsl:choose>
    <!-- Cross-book links: First test is for /publish=no, other 2 for /publish -->
    <xsl:when test="(key('xreflist',@linkend))[1]/self::nameloc or
                    @objtype='nameloc' or contains(@tgtids,' DOCID=')">
      <xsl:call-template name="output-required-cleanup-warning"/>
      <xsl:call-template name="process-l">
        <xsl:with-param name="cross-book">true</xsl:with-param>
      </xsl:call-template>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="process-l"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- P016846: If this is a cross-book link, it must go in required-cleanup,
              regardless of what other wrappers are created. Added $cross-book. -->
<xsl:template name="process-l">
  <xsl:param name="cross-book">false</xsl:param>
  <xsl:choose>
    <xsl:when test="parent::dbody or parent::notices or parent::ednotices or parent::safety or
                    parent::fig or parent::nitem">
      <xsl:choose>
        <xsl:when test="$cross-book='true'">
          <p><required-cleanup><xsl:call-template name="output-l-as-xref"/></required-cleanup></p>
        </xsl:when>
        <xsl:otherwise><p><xsl:call-template name="output-l-as-xref"/></p></xsl:otherwise>
      </xsl:choose>
    </xsl:when>
    <xsl:when test="parent::termhd or parent::defnhd">
      <xsl:choose>
        <xsl:when test="$cross-book='true'">
          <ph><required-cleanup><xsl:call-template name="output-l-as-xref"/></required-cleanup></ph>
        </xsl:when>
        <xsl:otherwise><ph><xsl:call-template name="output-l-as-xref"/></ph></xsl:otherwise>
      </xsl:choose>
    </xsl:when>
    <xsl:otherwise>
      <xsl:choose>
        <xsl:when test="$cross-book='true'">
          <required-cleanup><xsl:call-template name="output-l-as-xref"/></required-cleanup>
        </xsl:when>
        <xsl:otherwise><xsl:call-template name="output-l-as-xref"/></xsl:otherwise>
      </xsl:choose>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- ***************************** [13] PROC ****************************** -->

<!-- To make use of proc's standard structure, just output as task instead of
     using the output-divlike template. This allows easy mapping of many specific
     PROC elements. This also means that many proc elements will be mapped here,
     instead of within their own templates; the <title> tag is created here instead
     of in the proc/titleblk/title template. -->
<xsl:template name="process-proc">
  <task>
    <xsl:call-template name="add-language-attribute"/>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:value-of select="$newline"/>
    <title>
      <xsl:apply-templates select="@procnum"/>
      <xsl:apply-templates select="titleblk/title"/>
    </title>
    <xsl:if test="desc">
      <xsl:value-of select="$newline"/>
      <shortdesc>
        <xsl:apply-templates select="desc"/>
      </shortdesc>
    </xsl:if>
    <xsl:call-template name="output-topic-prolog"/>
    <xsl:value-of select="$newline"/>
    <taskbody>
      <xsl:apply-templates select="procentry"/>
      <xsl:if test="procsumm|procintro">
        <xsl:value-of select="$newline"/>
        <context>
          <xsl:apply-templates select="procsumm"/>
          <xsl:apply-templates select="procintro"/>
        </context>
      </xsl:if>
      <xsl:if test="procstep">
        <xsl:value-of select="$newline"/>
        <steps>
          <xsl:apply-templates select=".//procstep"/>
          <xsl:if test="procstep//procexit">
            <step>
              <xsl:attribute name="id">
                <xsl:value-of select="generate-id(.)"/><xsl:text>-end</xsl:text>
              </xsl:attribute>
              <cmd><xsl:value-of select="@end-text"/></cmd>
            </step>
          </xsl:if>
        </steps>
      </xsl:if>
      <xsl:apply-templates select="procexit"/>
    </taskbody>
    <!-- Could add a related-links section with prerequisites, based on @prereqprocs? -->
  </task>
</xsl:template>

<xsl:template match="proc/@procnum">
  <!-- output-message for procnum? or is this right? -->
  <xsl:value-of select="."/><xsl:text>:  </xsl:text>
</xsl:template>

<!-- As with other topics: if we are using Saxon, output a new file; otherwise, send
     the new topic to the output stream. -->
<xsl:template match="proc">
  <xsl:choose>
    <xsl:when test="@chunk-filename and contains(system-property('xsl:vendor'), 'SAXON')">
      <saxon:output href="{@chunk-filename}">
        <xsl:call-template name="use-task-dtd"/>
        <xsl:call-template name="process-proc"/>
      </saxon:output>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="process-proc"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="proc/titleblk/title">
  <xsl:apply-templates/>
</xsl:template>

<xsl:template match="proc/desc">
  <xsl:apply-templates/>
</xsl:template>  

<xsl:template match="proc/procentry">
  <prereq>
    <!-- output-message for pre-requisite procs from @relprocs, @prereqprocs, or support them -->
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates/>
    <xsl:if test="@prereqprocs">
    </xsl:if>
  </prereq>
</xsl:template>

<!-- In PDF, procsumm appears as a table at the beginning of a topic. We will place it in
     a table within the <context> element at the start of the task. -->
<xsl:template match="procsumm">
  <!-- Count the greatest number of items in a 'row', and use this as the number of columns -->
  <xsl:variable name="colsval">
    <xsl:apply-templates select="procsummitem[1]" mode="countcols"/>
  </xsl:variable>
  <xsl:value-of select="$newline"/>
  <table>
    <xsl:call-template name="add-default-attributes"/>
    <tgroup cols="{$colsval}">
      <xsl:value-of select="$newline"/>
      <tbody>
        <xsl:apply-templates select="procsummitem">
          <xsl:with-param name="mincols" select="number($colsval)"/>
        </xsl:apply-templates>
      </tbody>
    </tgroup>
    <xsl:value-of select="$newline"/>
  </table>
</xsl:template>

<!-- Determine the number of 'entries' in a procsummitem. Entries are procentry or
     procexit tags. The template counts the number in the current row; if there
     is a next row, the same template is called on the next row, and so on until
     all are checked. The largest number is returned. -->
<xsl:template match="procsummitem" mode="countcols">
  <xsl:param name="highcols" select="1"/> <!-- Current max number of entries -->
  <xsl:variable name="currentcols">       <!-- Number in the current row, or $highcols, -->
    <xsl:choose>                          <!-- whichever is greatest                    -->
      <xsl:when test="count(procentry|procexit) > number($highcols)">
        <xsl:value-of select="count(procentry|procexit)"/>
      </xsl:when>
      <xsl:otherwise><xsl:value-of select="$highcols"/></xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:choose>
    <!-- If there is another row, call the template on that row -->
    <xsl:when test="following-sibling::procsummitem">
      <xsl:apply-templates select="following-sibling::procsummitem[1]" mode="countcols">
        <xsl:with-param name="highcols"><xsl:value-of select="$currentcols"/></xsl:with-param>
      </xsl:apply-templates>
    </xsl:when>
    <!-- Otherwise, return the current maximum -->
    <xsl:otherwise><xsl:value-of select="$currentcols"/></xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Treat each procsummitem as a row. If the number of entries in this row is less
     than the max, fill out the table with empty entries. -->
<xsl:template match="procsummitem">
  <xsl:param name="mincols" select="1"/>
  <xsl:variable name="currentcols" select="count(procentry|procexit)"/>
  <xsl:value-of select="$newline"/>
  <row>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates/>
    <xsl:if test="$currentcols &lt; $mincols">
      <xsl:call-template name="add-entries">
        <xsl:with-param name="remaining" select="number(number($mincols) - number($currentcols))"/>
      </xsl:call-template>
    </xsl:if>
  </row>
</xsl:template>

<xsl:template match="procsummitem/procentry | procsummitem/procexit">
  <entry>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates/>
  </entry>
</xsl:template>

<xsl:template match="proc/procintro">
  <xsl:apply-templates/>
</xsl:template>

<!-- Turn each procstep into a step. Branches will link to steps below, instead of
     creating substeps. If the step has a title, place it in required-cleanup. -->
<xsl:template match="procstep">
  <xsl:value-of select="$newline"/>
  <step>
    <xsl:call-template name="add-default-attributes"/>
    <cmd>
      <xsl:if test="titleblk/title">
        <xsl:call-template name="output-required-cleanup-warning"/>
        <required-cleanup><xsl:apply-templates select="titleblk/title"/></required-cleanup>
      </xsl:if>
      <xsl:choose>
        <xsl:when test="proccmnd[not(child::procstep)][2]">  <!-- if there is more than one -->
          <!-- output-message? -->
          <xsl:for-each select="proccmnd[not(child::procstep)]">
            <ph><xsl:apply-templates select="."/></ph><xsl:text> </xsl:text>
          </xsl:for-each>
        </xsl:when>
        <xsl:otherwise><xsl:apply-templates select="proccmnd[not(child::procstep)]"/></xsl:otherwise>
      </xsl:choose>
    </cmd>
    <!-- if there was a proccmd/desc with block-like children, move them to <info> -->
    <xsl:if test="proccmnd/desc/ul|proccmnd/desc/dl|proccmnd/desc/ol|
                  proccmnd/desc/p|proccmnd/desc/pblk|proccmnd/desc/note|
                  proccmnd/desc/notelist|proccmnd/desc/fig">
      <info>
        <xsl:apply-templates select="(proccmnd/desc/text()|proccmnd/desc/*)[self::ul or preceding-sibling::ul or
                                 self::dl or preceding-sibling::dl or
                                 self::ol or preceding-sibling::ol or
                                 self::p or preceding-sibling::p or
                                 self::pblk or preceding-sibling::pblk or
                                 self::note or preceding-sibling::note or
                                 self::notelist or preceding-sibling::notelist or
                                 self::fig or preceding-sibling::fig]"/>
      </info>
    </xsl:if>
    <xsl:if test="stepnotes">
      <xsl:if test="proccmnd/following-sibling::stepnotes">
        <!-- output-message if stepnotes was moved after a command? -->
      </xsl:if>
      <xsl:apply-templates select="stepnotes"/>
    </xsl:if>
    <xsl:if test="decisionpnt">
      <xsl:apply-templates select="decisionpnt"/>
    </xsl:if>
    <xsl:if test="procexit">
      <xsl:apply-templates select="procexit"/>
    </xsl:if>
  </step>
</xsl:template>

<xsl:template match="procstep/titleblk/title">
  <xsl:apply-templates/>
</xsl:template>

<xsl:template match="proccmnd">
  <xsl:apply-templates/>
</xsl:template>


<xsl:template match="proccmnd/desc">
  <xsl:for-each select="(text()|*)[not(self::ul or preceding-sibling::ul or
                                       self::dl or preceding-sibling::dl or
                                       self::ol or preceding-sibling::ol or
                                       self::p or preceding-sibling::p or
                                       self::pblk or preceding-sibling::pblk or
                                       self::note or preceding-sibling::note or
                                       self::notelist or preceding-sibling::notelist or
                                       self::fig or preceding-sibling::fig)]">
    <xsl:apply-templates select="."/>
  </xsl:for-each>
</xsl:template>

<xsl:template match="stepnotes">
  <info>
    <xsl:call-template name="add-default-attributes"/>
    <ol><xsl:apply-templates/></ol>
  </info>
</xsl:template>

<xsl:template match="decisionpnt">
  <!-- If decisionpnt contains a <cond> element, place it in info -->
  <info><xsl:apply-templates select="cond"/></info>
  <!-- Otherwise, it should contain <then> and possible <else>. Call the <then> template,
       ane <else> will be applied from within -->
  <xsl:apply-templates select="then"/>
</xsl:template>

<!-- If "else" or "then" contain another procstep, that step is moved to a new <step>.
     This template will create a link to that step. 
     E003903: remove @type="li" from xref. -->
<xsl:template match="procstep" mode="see-step">
  <xsl:text>See </xsl:text>
  <xref>
    <xsl:attribute name="href">
      <xsl:text>#</xsl:text>
      <xsl:call-template name="get-current-topic-id"/>
      <xsl:text>/</xsl:text>
      <xsl:value-of select="@id"/>
    </xsl:attribute>
  </xref>
</xsl:template>

<!-- The <then> will always come before <else>, if there is one. So, call that template
     here, and place else in the proper spot in the choice table -->
<xsl:template match="then">
  <choicetable>
    <xsl:value-of select="$newline"/>
    <chrow>
      <xsl:call-template name="add-default-attributes"/>
      <choption><xsl:value-of select="@answertext"/></choption>
      <chdesc>
        <xsl:choose>
          <xsl:when test="procstep">
            <xsl:apply-templates select="procstep[1]" mode="see-step"/>
          </xsl:when>
          <xsl:otherwise><xsl:apply-templates/></xsl:otherwise>
        </xsl:choose>
      </chdesc>
    </chrow>
    <xsl:apply-templates select="following-sibling::else"/>
  </choicetable>
</xsl:template>

<xsl:template match="then/desc">
  <xsl:apply-templates/>
</xsl:template>

<xsl:template match="else">
  <xsl:value-of select="$newline"/>
  <chrow>
    <xsl:call-template name="add-default-attributes"/>
    <choption><xsl:value-of select="@answertext"/></choption>
      <chdesc>
        <xsl:choose>
          <xsl:when test="procstep">
            <xsl:apply-templates select="procstep[1]" mode="see-step"/>
          </xsl:when>
          <xsl:otherwise><xsl:apply-templates/></xsl:otherwise>
        </xsl:choose>
      </chdesc>
  </chrow>
</xsl:template>

<xsl:template match="else/desc">
  <xsl:apply-templates/>
</xsl:template>

<!-- Procexit is always applied at the end of the step, and should be placed in the step result.
     After the contents of the procexit, place a link to the end, so it is obvious that this
     ends the current path through the PROC.
     E003903: remove @type="li" from xref. -->
<xsl:template match="procstep/procexit">
  <stepresult>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates/>
    <xsl:text> </xsl:text><xsl:value-of select="ancestor::proc[1]/@see-text"/><xsl:text> </xsl:text>
    <xref>
      <xsl:attribute name="href">
        <xsl:text>#</xsl:text>
        <xsl:call-template name="get-current-topic-id"/>
        <xsl:text>/</xsl:text>
        <xsl:value-of select="generate-id(ancestor::proc[1])"/>
        <xsl:text>-end</xsl:text>
      </xsl:attribute>
    </xref>
  </stepresult>
</xsl:template>

<!-- The procexit for the entire procedure can be placed in <result>. -->
<xsl:template match="proc/procexit">
  <xsl:value-of select="$newline"/>
  <result>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates/>
  </result>
</xsl:template>

<!-- Create an xref to the referenced step. E003903: remove @type="li" from xref. -->
<xsl:template match="stepref">
  <xsl:variable name="thisfile"><xsl:call-template name="get-current-chunk-name"/></xsl:variable>
  <xref>
    <xsl:attribute name="href">
      <xsl:apply-templates select="key('xreflist',@refid)[1]" mode="get-xref-target">
        <xsl:with-param name="xreffile"><xsl:value-of select="$thisfile"/></xsl:with-param>
      </xsl:apply-templates>
    </xsl:attribute>
  </xref>
</xsl:template>

<!-- ********************** [14] IMAGES, IMAGEDESCS ************************ -->

<!-- In many places in DITA, images need an additional wrapper. Create the wrapper
     here, and call a standard template that will create the image. 
     There are 2 standard templates: one creates a basic image, and one uses
     the mmobjlink to create a link around the image. -->
<xsl:template match="mmobj">
  <xsl:param name="parent-element"/>
  <xsl:apply-templates select="i1|i2|i3|iref" mode="ensure-proper-placement">
    <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
  </xsl:apply-templates>
  <xsl:choose>                   <!-- parent=bridge, copyr, mknote, nitem, screen? -->
    <xsl:when test="mmobjlink and $parent-element='refbody'">
      <section><xsl:call-template name="output-linked-image"/></section>
    </xsl:when>
    <xsl:when test="mmobjlink and $parent-element='conbody' and preceding-sibling::pblk">
      <section><xsl:call-template name="output-linked-image"/></section>
    </xsl:when>
    <xsl:when test="mmobjlink and $parent-element='screen'">
      <ph><xsl:call-template name="output-linked-image"/></ph>
    </xsl:when>
    <xsl:when test="mmobjlink and (parent::dbody or parent::ednotices or parent::notices or
                                   parent::partasm or parent::partasmseg or parent::safety)">
      <p><xsl:call-template name="output-linked-image"/></p>
    </xsl:when>
    <xsl:when test="mmobjlink">
      <xsl:call-template name="output-linked-image"/>
    </xsl:when>
    <xsl:when test="$parent-element='refbody'">
      <section><xsl:call-template name="output-image"/></section>
    </xsl:when>
    <xsl:when test="$parent-element='conbody' and preceding-sibling::pblk">
      <section><xsl:call-template name="output-image"/></section>
    </xsl:when>
    <xsl:when test="$parent-element='screen'">
      <ph><xsl:call-template name="output-image"/></ph>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="output-image"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Output an image, with an XREF around it. This template is called when the image has an
     mmobjlink element.
     E003903: remove @type from xref. It should be picked up during transforms. -->
<xsl:template name="output-linked-image">
  <xsl:variable name="target"><xsl:call-template name="determine-link-target"/></xsl:variable>
  <xsl:variable name="thisfile"><xsl:call-template name="get-current-chunk-name"/></xsl:variable>
  <xsl:variable name="scope">
    <xsl:choose>
      <xsl:when test="$target='url' and key('xreflist',@linkend)/../@class='newwindow'">external</xsl:when>
      <xsl:when test="$target='url' and key('xreflist',@linkend)/../@class='samewindow'">local</xsl:when>
      <xsl:when test="$target='url' and key('xreflist',@linkend)/../@class='fullwindow'">local</xsl:when>
    </xsl:choose>
  </xsl:variable>
  <xref>
    <!--<xsl:if test="string-length($target)>0">
      <xsl:attribute name="type"><xsl:value-of select="$target"/></xsl:attribute>
    </xsl:if>-->
    <xsl:if test="string-length($scope)>0">
      <xsl:attribute name="scope"><xsl:value-of select="$scope"/></xsl:attribute>
    </xsl:if>
    <xsl:attribute name="href">
      <xsl:apply-templates select="key('xreflist',mmobjlink/@linkend)[1]" mode="get-xref-target">
        <xsl:with-param name="xreffile"><xsl:value-of select="$thisfile"/></xsl:with-param>
      </xsl:apply-templates>
    </xsl:attribute>
    <xsl:call-template name="output-image"/>
  </xref>
</xsl:template>

<!-- Output an image, saving as many attributes as possible. -->
<xsl:template name="output-image">
  <image>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:attribute name="placement">
      <xsl:choose>
        <xsl:when test="@placement='inline'">inline</xsl:when>
        <xsl:when test="@placement='standalone'">break</xsl:when>
        <xsl:when test="@placement='margin'">break</xsl:when> <!-- output-message: unsupported attr value -->
        <xsl:otherwise>break</xsl:otherwise>
      </xsl:choose>
    </xsl:attribute>
    <xsl:apply-templates select="@halign"/>
    <xsl:apply-templates select="@longdescid"/>
    <xsl:apply-templates select="objref"/>
    <xsl:apply-templates select="textalt"/>  <!-- Will combine with @label if both specified -->
    <xsl:if test="not(textalt)"><xsl:apply-templates select="@label"/></xsl:if>
  </image>
</xsl:template>

<xsl:template match="mmobj/@halign">
  <xsl:if test="not(contains(@halign,'current'))">
    <xsl:attribute name="align"><xsl:value-of select="."/></xsl:attribute>
  </xsl:if>
</xsl:template>

<xsl:template match="mmobj/@longdescid">
  <xsl:attribute name="longdescref">
    <!-- Set the attribute to the file name that the referenced -->  
    <!-- londesc section was mapped to -->
    <xsl:value-of select="key('xreflist',.)[1]/@chunk-filename"/>
  </xsl:attribute>
</xsl:template>

<xsl:template match="mmobj/@label">
  <alt><xsl:value-of select="normalize-space(.)"/></alt>
</xsl:template>

<!-- E003331: image dimensions should be converted to pixel values, if possible. -->
<xsl:template name="convert-to-pixel">
  <xsl:param name="inputval"/>
  <xsl:choose>
    <xsl:when test="contains($inputval,'in') or contains($inputval,'cm') or
                    contains($inputval,'mm') or contains($inputval,'pt') or contains($inputval,'pi')">
      <!-- Determine the pixel value, including any decimal -->
      <xsl:variable name="pixels-plus-decimal">
        <xsl:choose>
          <xsl:when test="contains($inputval,'in')">
            <xsl:value-of select="number(substring-before($inputval,'in')) * 96"/>
          </xsl:when>
          <xsl:when test="contains($inputval,'cm')">
            <xsl:value-of select="(number(substring-before($inputval,'cm')) div 2.54) * 96"/>
          </xsl:when>
          <xsl:when test="contains($inputval,'mm')">
            <xsl:value-of select="(number(substring-before($inputval,'mm')) div 25.4) * 96"/>
          </xsl:when>
          <xsl:when test="contains($inputval,'pt')">
            <xsl:value-of select="(number(substring-before($inputval,'pt')) * 96) div 72"/>
          </xsl:when>
          <xsl:when test="contains($inputval,'pi')">
            <xsl:variable name="points">
              <xsl:choose>
                <xsl:when test="contains($inputval,'.')">
                  <xsl:value-of select="substring-after(substring-before($inputval,'pi'),'.')"/>
                </xsl:when>
                <xsl:otherwise>0</xsl:otherwise>
              </xsl:choose>
            </xsl:variable>
            <xsl:variable name="picas">
              <xsl:choose>
                <xsl:when test="contains($inputval,'.')"><xsl:value-of select="substring-before($inputval,'.')"/></xsl:when>
                <xsl:otherwise><xsl:value-of select="substring-before($inputval,'pi')"/></xsl:otherwise>
              </xsl:choose>
            </xsl:variable>
            <xsl:value-of select="(((number($picas) * 12) + number($points)) * 96) div 72"/>
          </xsl:when>
        </xsl:choose>
      </xsl:variable>
      <!-- Round down to the nearest pixel (decimals are silly with pixels) -->
      <xsl:choose>
        <xsl:when test="contains($pixels-plus-decimal,'.')">
          <xsl:value-of select="substring-before($pixels-plus-decimal,'.')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="$pixels-plus-decimal"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>
    <xsl:otherwise>
      <xsl:value-of select="$inputval"/>
    </xsl:otherwise>
  </xsl:choose>                
</xsl:template>

<!-- Use the objref element to crete the HREF, and save the width/depth attributes. -->
<xsl:template match="objref">
  <!-- output-message for @scalebox or @editscale -->
  <xsl:apply-templates select="@width"/>
  <xsl:apply-templates select="@depth"/>
  <xsl:attribute name="href">
    <xsl:value-of select="@href"/>
  </xsl:attribute>
</xsl:template>

<xsl:template match="objref/@width">
  <xsl:attribute name="width">
    <xsl:call-template name="convert-to-pixel">
      <xsl:with-param name="inputval"><xsl:value-of select="."/></xsl:with-param>
    </xsl:call-template>
  </xsl:attribute>
</xsl:template>

<xsl:template match="objref/@depth">
  <xsl:attribute name="height">
    <xsl:call-template name="convert-to-pixel">
      <xsl:with-param name="inputval"><xsl:value-of select="."/></xsl:with-param>
    </xsl:call-template>
  </xsl:attribute>
</xsl:template>

<!-- Use value-of to prevent saving phrase markup, because we are moving the content
     from an element to an attribute. -->
<xsl:template match="textalt">
  <alt>
    <xsl:if test="../@label">
      <xsl:value-of select="normalize-space(../@label)"/><xsl:text> </xsl:text>
    </xsl:if>
    <xsl:value-of select="normalize-space(.)"/>
  </alt>
</xsl:template>

<!-- This is not yet supported in any transform, much less in DITA, so it should not be
     in documents. -->
<xsl:template match="areadef">
  <xsl:call-template name="unsupported-element"/>
</xsl:template>

<!-- This is just a container for the div-like longdesc elements. -->
<xsl:template match="imagedescs">
  <xsl:apply-templates/>
</xsl:template>

<!-- Turn each longdesc into a simple topic. -->
<xsl:template match="longdesc">
  <xsl:call-template name="output-divlike"/>
</xsl:template>


<!-- ************************ [15] ASMList, PartASM ************************ -->

<!-- Asmlist is a list of all the partasm's in a document; if they really want that
     list, we may as well search using //. Should not occur too often, and our
     documentation says you should have a max of one per book. -->
<xsl:template match="asmlist">
  <xsl:value-of select="$newline"/>
  <ul>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates select="//partasm/title" mode="asmlist"/>
  </ul>
</xsl:template>

<xsl:template match="partasm/title" mode="asmlist">
  <xsl:value-of select="$newline"/>
  <li>
    <xsl:apply-templates/>
  </li>
</xsl:template>

<!-- Output a new topic for partasm. Default to 'reference' as the type. -->
<xsl:template match="partasm">
  <xsl:call-template name="output-divlike">
    <xsl:with-param name="default-type">reference</xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!-- If @assemblytitle is specified, merge that with the title contents. The title in
     PDF contains a sequential number for each partasm in the book, so add that too. -->
<xsl:template match="partasm/title">
  <xsl:value-of select="$newline"/>
  <title>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:value-of select="../@assemblytitle"/>
    <xsl:text> </xsl:text><xsl:number count="partasm" level="any"/><xsl:text>: </xsl:text>
    <xsl:apply-templates/>
  </title>
</xsl:template>

<!-- No equivalent in DITA to save the partasmseg tag. -->
<xsl:template match="partasmseg">
  <xsl:apply-templates/>
</xsl:template>

<!-- Turn the <compl> into a table, which visually matches the PDF output. Headings are
     added as attributes by the id2xid pre-processor.
     Each <ci> in the list creates a row. With nested <compl> elements, process the
     children. -->
<xsl:template match="compl">
  <xsl:value-of select="$newline"/>
  <table>
    <xsl:call-template name="add-default-attributes"/>
    <tgroup cols="4">
      <xsl:value-of select="$newline"/>
      <thead>
        <row>
          <entry><xsl:value-of select="ancestor::partasm[1]/@asmindextitle"/></entry>
          <entry><xsl:value-of select="ancestor::partasm[1]/@partnumtitle"/></entry>
          <entry><xsl:value-of select="ancestor::partasm[1]/@unitstitle"/></entry>
          <entry><xsl:value-of select="ancestor::partasm[1]/@desctitle"/></entry>
        </row>
      </thead>
      <xsl:value-of select="$newline"/>
      <tbody>
        <xsl:apply-templates select="ci|compl"/>
      </tbody>
    </tgroup>
    <xsl:if test="@conloc">
      <tgroup cols="4"><tbody><row><entry/><entry/><entry/><entry/></row></tbody></tgroup>
    </xsl:if>
  </table>
</xsl:template>

<!-- Process any compl elements that are already within a compl. The <ci> elements should
     be processed and come out as rows. -->
<xsl:template match="compl/compl">
  <xsl:apply-templates select="ci|compl"/>
</xsl:template>

<!-- Map CI to a row.
     * First column: @idxnum (combined with either a dash, or the number of this partasm)
     * Second: @partnum
     * Third: @upa (units)
     * Fourth: CI contents, and any compcmt that comes between this CI and the next --> 
<xsl:template match="ci">
  <xsl:value-of select="$newline"/>
  <row>
    <xsl:call-template name="add-default-attributes"/>
    <!-- If this is the first CI, use the partasm number + dash + @idxnum.
         Otherwise, use a dash + @idxnum. -->
    <entry>
      <xsl:choose>
        <xsl:when test="preceding-sibling::ci or parent::compl/preceding-sibling::ci">
          <xsl:text>-</xsl:text><xsl:value-of select="@idxnum"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:number count="partasm" level="any"/>-<xsl:value-of select="@idxnum"/>
        </xsl:otherwise>
      </xsl:choose>
    </entry>
    <!-- Output this part number, or a space if none is present -->
    <entry>
      <xsl:choose>
        <xsl:when test="@partnum"><xsl:value-of select="@partnum"/></xsl:when>
        <xsl:otherwise>&nbsp;</xsl:otherwise>
      </xsl:choose>
    </entry>
    <!-- Output the number of units, or a space if unspecified -->
    <entry>
      <xsl:choose>
        <xsl:when test="@upa"><xsl:value-of select="@upa"/></xsl:when>
        <xsl:otherwise>&nbsp;</xsl:otherwise>
      </xsl:choose>
    </entry>
    <!-- Output the contents of the CI, and following compcmt elements. -->
    <entry>
      
      <!-- If this is in a nested compl, the CI should have a bullet. -->
      <xsl:choose>
        <xsl:when test="parent::compl/parent::compl">
          <ul><li><xsl:apply-templates/></li></ul>
        </xsl:when>
        <xsl:otherwise>
          <xsl:apply-templates/>
        </xsl:otherwise>
      </xsl:choose>

      <!-- Find the position of the following CI or COMPL. Apply templates on any compcmt elements
           which appear after the current CI, and before the next COMPL or CI. -->
      <xsl:choose>
        <xsl:when test="following-sibling::ci">
          <xsl:variable name="ciposition">
            <xsl:for-each select="following-sibling::*[self::ci or self::compl][1]">
              <xsl:value-of select="count(preceding-sibling::*)+1"/>
            </xsl:for-each>
          </xsl:variable>
          <xsl:for-each select="following-sibling::compcmt">
            <!-- Cannot use position(), because the node set is only the set of following compcmt's -->
            <xsl:if test="count(preceding-sibling::*)+1 &lt; number($ciposition)">
              <xsl:apply-templates select="."/>
            </xsl:if>
          </xsl:for-each>
        </xsl:when>
        <xsl:otherwise>
          <xsl:apply-templates select="following-sibling::compcmt"/>
        </xsl:otherwise>
      </xsl:choose>

      <!-- If this is the end of a nested compl, and the nested compl is followed by compcmt
           elements, output those here. -->
      <xsl:if test="parent::compl/parent::compl and 
                    not(following-sibling::ci) and
                    ../following-sibling::compcmt">
        <!-- Get the position of the first CI after this nested COMPL -->
        <xsl:variable name="ciposition">
          <xsl:for-each select="../following-sibling::ci[1]">
            <xsl:value-of select="count(preceding-sibling::*)+1"/>
          </xsl:for-each>
        </xsl:variable>

        <xsl:choose>
          <!-- If there are no CI elements after this compl, process all following compcmt elements -->
          <xsl:when test="not(../following-sibling::ci)">
            <xsl:apply-templates select="../following-sibling::compcmt"/>
          </xsl:when>
          <!-- Process following compcmt, which occur before the next CI -->
          <xsl:otherwise>
            <xsl:apply-templates select="../following-sibling::compcmt[count(preceding-sibling::*)+1 &lt; number($ciposition)]"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:if>

    </entry>
  </row>
</xsl:template>


<xsl:template match="compcmt">
  <xsl:text> </xsl:text>
  <ph><xsl:apply-templates/></ph>
</xsl:template>

<!-- ************************* [16] MESSAGE LISTS ************************** -->

<!-- The message list element is ignored. Each message creates a topic, so there is nothing
     for the msglist element to do.
     Theoretically we could turn the msglist into a <dl>, but there is no way to link to a
     single dlentry element. -->
<xsl:template match="msglist">      <!-- output-message for conloc? -->
  <xsl:choose>
    <xsl:when test="$MSGLIST='newfile'">
      <xsl:apply-templates/>
    </xsl:when>
    <xsl:otherwise>
      <xsl:if test="$MSGLIST!='dl'">
        <!-- output-message -->
      </xsl:if>
      <xsl:value-of select="$newline"/>
      <dl>
        <xsl:call-template name="add-default-attributes"/>
        <xsl:apply-templates/>
        <xsl:value-of select="$newline"/>
      </dl>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- For now, the only option is to turn each msg into a new file. The default type is reference. -->
<xsl:template match="msg">
  <xsl:choose>
    <xsl:when test="$MSGLIST='newfile' and @chunk-filename and 
                    contains(system-property('xsl:vendor'), 'SAXON')">
      <saxon:output href="{@chunk-filename}">
        <xsl:call-template name="use-reference-dtd"/>
        <xsl:call-template name="output-msg-as-reference"/>
      </saxon:output>
    </xsl:when>
    <xsl:when test="$MSGLIST='newfile'">
      <xsl:call-template name="output-msg-as-reference"/>
    </xsl:when>
    <xsl:otherwise>
      <xsl:value-of select="$newline"/>
      <dlentry>
        <xsl:call-template name="add-default-attributes"/>
        <dt><xsl:call-template name="output-msg-title"/></dt>
        <xsl:apply-templates select="msgitem"/>
      </dlentry>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Any index entries at the start of the msg are pulled into the keywords.
     Other than that, metadata is currently ignored for messages. -->
<xsl:template name="output-msg-as-reference">
  <reference>
    <xsl:call-template name="add-language-attribute"/>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:value-of select="$newline"/>
    <title><xsl:call-template name="output-msg-title"/></title>
    <xsl:if test="i1|i2|i3|iref|@otherprops">
      <xsl:value-of select="$newline"/>
      <prolog><metadata>
        <xsl:if test="i1|i2|i3|iref">
          <keywords>
            <!-- Output entries as keywords, if they appear before or after all msgitem elements.
                 Entries between msgitems are assumed to apply only to the next msgitem. -->
            <xsl:apply-templates select="i1|i2|i3|iref" mode="ensure-proper-placement"/>
          </keywords>
        </xsl:if>
        <!--<xsl:call-template name="output-otherprops-as-othermeta"/>-->
        <xsl:value-of select="$newline"/>
      </metadata></prolog>
    </xsl:if>
    <xsl:value-of select="$newline"/>
    <refbody>
      <xsl:apply-templates select="msgitem"/>
    </refbody>
  </reference>
</xsl:template>

<!-- The message title is made up of either code, or some combination of msgnum and msgtext -->
<xsl:template name="output-msg-title">
  <xsl:choose>
    <xsl:when test="code"><xsl:apply-templates select="code"/></xsl:when>
    <xsl:when test="msgnum and msgtext">
      <ph><xsl:apply-templates select="msgnum"/></ph>
      <xsl:text>: </xsl:text>
      <ph><xsl:apply-templates select="msgtext"/></ph>
    </xsl:when>
    <xsl:when test="msgnum"><xsl:apply-templates select="msgnum"/></xsl:when>
    <xsl:when test="msgtext"><xsl:apply-templates select="msgtext"/></xsl:when>
  </xsl:choose>
</xsl:template>

<!-- Default titles for msgitem are passed down on msgitemdef elements by the id2xid pre-processor -->
<xsl:template name="get-msgitem-title">
  <xsl:choose>
    <xsl:when test="../../msgitemdef[@classname=current()/@class]">
      <xsl:apply-templates select="../../msgitemdef[@classname=current()/@class]"
                           mode="output-msgitem-title"/>
    </xsl:when>
    <xsl:when test="ancestor::*/dprolog/propdefs/msgitemdef[@classname=current()/@class] |
                    ancestor::*/specdprolog/propdefs/msgitemdef[@classname=current()/@class]">
      <xsl:apply-templates select="(ancestor::*/dprolog/propdefs/msgitemdef[@classname=current()/@class] |
                                    ancestor::*/specdprolog/propdefs/msgitemdef[@classname=current()/@class])[last()]"
                           mode="output-msgitem-title"/>
    </xsl:when>
    <xsl:when test="/ibmiddoc/prolog/propdefs/msgitemdef[@classname=current()/@class]">
      <xsl:apply-templates select="/ibmiddoc/prolog/propdefs/msgitemdef[@classname=current()/@class]"
                           mode="output-msgitem-title"/>
    </xsl:when>
    <xsl:otherwise><xsl:value-of select="@class"/></xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Currently, new file is the only option. So, msgitem maps to section; the default titles
     are placed in the section title. The <dd> mapping is a rough guess at future possibilities. -->
<xsl:template match="msgitem">
  <xsl:choose>
    <xsl:when test="$MSGLIST='newfile'">
      <xsl:value-of select="$newline"/>
      <section>
        <xsl:variable name="testtitle"><xsl:call-template name="get-msgitem-title"/></xsl:variable>
        <xsl:call-template name="add-default-attributes"/>
        <xsl:if test="string-length(normalize-space($testtitle))!=0">
          <title><xsl:call-template name="get-msgitem-title"/></title>
        </xsl:if>
        <xsl:apply-templates>
          <xsl:with-param name="parent-element">section</xsl:with-param>
        </xsl:apply-templates>
      </section>
    </xsl:when>
    <xsl:otherwise>
      <xsl:value-of select="$newline"/>
      <dd>
        <dl>  <!-- would like to map to section within dl, but that is only available at <body> level -->
          <xsl:call-template name="add-default-attributes"/>
          <dlentry>
            <dt><xsl:call-template name="get-msgitem-title"/></dt>
            <dd><xsl:apply-templates/></dd>
          </dlentry>
        </dl>
      </dd>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- The msgitemdef and its title should not appear for fallthrough processing. -->
<xsl:template match="msgitemdef" mode="output-msgitem-title">
  <xsl:apply-templates/>
</xsl:template>
<xsl:template match="msgitemdef/title">
  <xsl:apply-templates/>
</xsl:template>

<!-- **************************** [17] INDEXING **************************** -->

<!-- Many locations in DITA do not allow index entries, which is a problem because they
     are allowed anywhere in IDDoc. So, in many locations the entries must be collected and
     moved to a valid location. This sometimes requires an extra phrase or section, but
     other times it requires a required-cleanup container. 
     When an index entry is not valid, the fallthrough processing will not work. To move it,
     we re-process the entry with mode=ensure-proper-placement. -->

<!-- Store any index entries which have IDs, for quick lookup by IREFs or other entries. -->
<xsl:key name="i1list" match="i1[@id]" use="@id"/>
<xsl:key name="i2list" match="i2[@id]" use="@id"/>
<xsl:key name="i3list" match="i3[@id]" use="@id"/>

<!-- [17a] Functions for working with index entries -->

<!-- Determine what wrapper, if any, is needed for index entries at the current location. -->
<xsl:template name="find-i1-wrapper">
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <xsl:when test="$parent-element='refbody' or
                    ($parent-element='conbody' and preceding-sibling::pblk)">section</xsl:when>
    <!-- Entries in task are always valid, so catch them before the next dbody test. -->
    <xsl:when test="$parent-element='context' or $parent-element='result'"/> 
    <!-- Entries at the start of dbody will be in keywords, without any wrapper. Entries in the
         middle of dbody will be in place, inside required-cleanup. This test to see if there
         are any non-index elements before this index, when parent is dbody. -->
    <xsl:when test="parent::dbody and preceding-sibling::*[not(name()='i1' or name()='i2' or
                                                               name()='i3' or name()='iref')]">
    
      <xsl:call-template name="output-required-cleanup-warning"/>
      <xsl:text>required-cleanup</xsl:text>
    </xsl:when>
    <xsl:when test="parent::ol or parent::ul">
      <xsl:call-template name="output-required-cleanup-warning"/>
      <xsl:text>required-cleanup</xsl:text>
    </xsl:when>
    <xsl:when test="parent::dl or parent::parml or parent::gl">
      <xsl:call-template name="output-required-cleanup-warning"/>
      <xsl:text>required-cleanup</xsl:text>
    </xsl:when>
    <!-- When parent is fig, wrapper is created in "fig" template -->
    <xsl:when test="parent::title">ph</xsl:when>
    <xsl:when test="parent::len">ph</xsl:when>
    <xsl:when test="parent::cap">ph</xsl:when>
    <xsl:when test="parent::fn">ph</xsl:when>
    <xsl:when test="parent::termhd or parent::defnhd">ph</xsl:when>
    <xsl:when test="parent::code">ph</xsl:when>
    <xsl:when test="parent::msgnum and not(../following-sibling::msgtext)">ph</xsl:when>
    <xsl:when test="parent::msgtext and not(../preceding-sibling::msgnum)">ph</xsl:when>
    <xsl:when test="parent::ph[@style][@style!='base' and @style!='smallcaps']">ph</xsl:when>
    <xsl:when test="parent::l">ph</xsl:when>
    <xsl:when test="parent::term">ph</xsl:when>
    <xsl:when test="parent::annot or parent::annotbody">ph</xsl:when>
    <xsl:when test="parent::nitem">
      <xsl:call-template name="output-required-cleanup-warning"/>
      <xsl:text>required-cleanup</xsl:text>
    </xsl:when>
  </xsl:choose>
</xsl:template>

<!-- This determines if the index entry is in a valid location. If it is, we can output it.
     If not, it will be called elsewhere using the ensure-proper-placement mode. -->
<xsl:template name="inside-valid-index-location">
  <xsl:choose>
    <xsl:when test="parent::le or parent::lers or parent::d or parent::part or
                    parent::ol or parent::ul or parent::notelist or
                    parent::dl or parent::parml or parent::gl or
                    parent::dlentry or parent::parm or parent::glentry or
                    parent::table or parent::tgroup or parent::tbody or
                    parent::thead or parent::tfoot or parent::row or
                    parent::fig or parent::mv or parent::pk or parent::pv or
                    parent::q or parent::term or parent::xph or parent::synph or ancestor::syntax or
                    parent::var or parent::sep or parent::oper or parent::kwd or parent::delim or
                    parent::abbrev or parent::bibliog or parent::glossary or
                    parent::legend or parent::preface or parent::safety or
                    parent::soa or parent::abstract/parent::frontm or 
                    parent::notices or parent::ednotices or parent::mod">
      <xsl:text>invalid</xsl:text>
    </xsl:when>
    <!-- Entries in the dbody of an introductory abstract are valid, because the content will already
         be in a required-cleanup element. -->
    <xsl:when test="parent::dbody/parent::abstract/parent::*[not(name()='frontm')]">
      <xsl:text>valid</xsl:text>
    </xsl:when>
    <!-- Entries in dbody are only valid for fallthrough if there is other content in the dbody
         before the entry. If they come first in the division, they will be pulled to <keywords>. -->
    <xsl:when test="parent::dbody and preceding-sibling::*[not(name()='i1' or name()='i2' or
                                                               name()='i3' or name()='iref')]">valid</xsl:when>
    <xsl:when test="parent::dbody">invalid</xsl:when>
    
    <!-- Entries in a msg that are outside of any <msgitem> will be pulled into the <keywords>.
         Note: this is only true when each message is a new topic. -->
    <xsl:when test="parent::msg and $MSGLIST='newfile'">
      <xsl:text>invalid</xsl:text>
    </xsl:when>
    
    <xsl:when test="parent::msglist and $MSGLIST='newfile'">
      <xsl:text>invalid</xsl:text>
    </xsl:when>
    <xsl:when test="parent::frontm or parent::backm">
      <!-- output-message: cannot save index entry. -->
      <xsl:text>invalid</xsl:text>
    </xsl:when>
    <xsl:otherwise>
      <xsl:text>valid</xsl:text>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- [17b] Templates for processing inline index entries -->

<!-- If an index term goes into <keywords> put out a newline, to help keep string length down -->
<xsl:template name="output-newline-before-indexterm">
  <xsl:if test="parent::d or parent::lers or ancestor::dprolog or ancestor::specdprolog">
    <xsl:value-of select="$newline"/>
  </xsl:if>
</xsl:template>

<!-- Process an i1 entry that appears inline. Each idxterm will create an indexterm, with
     all i2 and i3 elements copied into each indexterm. If the i1 has an ID, it will be
     placed on the first indexterm. -->
<xsl:template name="process-i1-entry">
  <xsl:for-each select="idxterm">
    <indexterm>
      <xsl:if test="position()='1' and parent::i1/@id">
        <xsl:attribute name="id"><xsl:value-of select="parent::i1/@id"/></xsl:attribute>
      </xsl:if>
      <xsl:value-of select="."/>
      <xsl:apply-templates select="parent::i1/i2"/>
    </indexterm>
  </xsl:for-each>
</xsl:template>

<!-- This is called when we know we want to create the inline index entry. If a wrapper element
     is needed, create it and process the entry; otherwise just process the entry. -->
<xsl:template match="i1" mode="ensure-proper-placement">
  <xsl:param name="parent-element"/>
  <xsl:param name="wrapper">
    <xsl:call-template name="find-i1-wrapper">
      <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
    </xsl:call-template>
  </xsl:param>
  <xsl:call-template name="output-newline-before-indexterm"/>
  <xsl:choose>
    <!-- Entries in refbody, or after a section in conbody, must be in section to be in context. But, they
         should still be in a required-cleanup element. -->
    <xsl:when test="($parent-element='refbody' or $parent-element='conbody') and $wrapper='section'">
      <xsl:call-template name="output-required-cleanup-warning"/>
      <section><required-cleanup><xsl:call-template name="process-i1-entry"/></required-cleanup></section>
    </xsl:when>
    <xsl:when test="string-length($wrapper)>0">
      <xsl:element name="{$wrapper}">
        <xsl:call-template name="process-i1-entry"/>
      </xsl:element>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="process-i1-entry"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- First check to see if the current location is valid for fallthrough index processing.
     If so, create the index entry. If not, ignore it for now. -->
<xsl:template match="i1">
  <xsl:param name="parent-element"/>
  <xsl:variable name="usei1"><xsl:call-template name="inside-valid-index-location"/></xsl:variable>
  <xsl:if test="$usei1='valid'">
    <xsl:apply-templates select="." mode="ensure-proper-placement">
      <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
    </xsl:apply-templates>
  </xsl:if>
</xsl:template>

<!-- This is called when we know we want to create the inline index entry. First determine
     if a wrapper element is needed (although it will be created by the i1, not the i2).
     * If the i2 stands alone (it references an i1 with @i1id), save the value of this and
       any i3 elements, and process the referenced i1.
     * Otherwise, the the i2 is inside an i1, so create the nested indexterm and process children
     
     NOTE: if i1id is used, the referenced i1 is processed with a special mode, because its 
           location and children are not important (only the value).
      -->
<xsl:template match="i2" mode="ensure-proper-placement">
  <xsl:param name="parent-element"/>
  <xsl:param name="wrapper">   <!-- Find the wrapper element. This is skipped if we are in an i1. -->
    <xsl:if test="not(parent::i1)">
      <xsl:call-template name="find-i1-wrapper">
        <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
      </xsl:call-template>
    </xsl:if>
  </xsl:param>
  <xsl:choose>
    <!-- When the parent i1 is located elsewhere, and there are i3 elements:
         Save the value of the i2 and the wrapper. For each i3, process the referenced
         i1 tag, and pass the wrapper, i2 value, and i3 value. Each i3 will create one
         3-level set of indices (or more, if the i1 has many idxterms). -->
    <xsl:when test="@i1id and i3">
      <xsl:variable name="i2val" select="idxterm"/>
      <xsl:for-each select="i3">
        <xsl:apply-templates select="key('i1list',parent::i2/@i1id)" mode="index-by-reference">
          <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
          <xsl:with-param name="wrapper"><xsl:value-of select="$wrapper"/></xsl:with-param>
          <xsl:with-param name="i2val"><xsl:value-of select="$i2val"/></xsl:with-param>
          <xsl:with-param name="i3val" select="idxterm"/>
        </xsl:apply-templates>
      </xsl:for-each>
    </xsl:when>
    <!-- When the parent i1 is located elsewhere, and there are no i3 elements:
         Save the value of this i2, and process the referenced i1. This will create
         one 2-level set of terms for each idxterm in the referenced i1. -->
    <xsl:when test="@i1id">
      <xsl:apply-templates select="key('i1list',@i1id)" mode="index-by-reference">
        <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
        <xsl:with-param name="wrapper"><xsl:value-of select="$wrapper"/></xsl:with-param>
        <xsl:with-param name="i2val" select="idxterm"/>
      </xsl:apply-templates>
    </xsl:when>
    <!-- The i2 is located within an i1, so just process it and move on. -->
    <xsl:otherwise>
      <indexterm>
        <xsl:call-template name="add-default-attributes"/>
        <xsl:value-of select="idxterm"/>
        <xsl:apply-templates select="i3"/>
      </indexterm>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- First check to see if the current location is valid for fallthrough index processing.
     If so, create the index entry. If not, ignore it for now. -->
<xsl:template match="i2">
  <xsl:param name="parent-element"/>
  <xsl:variable name="usei2"><xsl:call-template name="inside-valid-index-location"/></xsl:variable>
  <xsl:if test="$usei2='valid'">
    <xsl:apply-templates select="." mode="ensure-proper-placement">
      <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
    </xsl:apply-templates>
  </xsl:if>
</xsl:template>

<!-- This is called when we know we want to create the inline index entry. First determine
     if a wrapper element is needed (although it will be created by the i1, not the i3).
     Processing is similar to i2. -->
<xsl:template match="i3" mode="ensure-proper-placement">
  <xsl:param name="parent-element"/>
  <xsl:param name="wrapper">    <!-- Don't waste time finding a wrapper if parent is i2 -->
    <xsl:if test="not(parent::i2)">
      <xsl:call-template name="find-i1-wrapper">
        <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
      </xsl:call-template>
    </xsl:if>
  </xsl:param>
  <xsl:choose>
    <!-- If this references an i2, save the value of this term, and process the referenced i2.
         This will create one 3-level set of terms for each idxterm in the ancestor i1. 
         Also pass the wrapper as a parameter: the wrapper must be determined by the
         current location, not by the location of a referenced i1 or i2. -->
    <xsl:when test="@i2id">
      <xsl:apply-templates select="key('i2list',@i2id)" mode="index-by-reference">
        <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
        <xsl:with-param name="wrapper"><xsl:value-of select="$wrapper"/></xsl:with-param>
        <xsl:with-param name="i3val" select="idxterm"/>
      </xsl:apply-templates>
    </xsl:when>
    <!-- Otherwise, the parent is an i2, so just create the term. -->
    <xsl:otherwise>
      <indexterm>
        <xsl:call-template name="add-default-attributes"/>
        <xsl:value-of select="idxterm"/>
      </indexterm>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- First check to see if the current location is valid for fallthrough index processing.
     If so, create the index entry. If not, ignore it for now. -->
<xsl:template match="i3">
  <xsl:param name="parent-element"/>
  <xsl:variable name="usei3"><xsl:call-template name="inside-valid-index-location"/></xsl:variable>
  <xsl:if test="$usei3='valid'">
    <xsl:apply-templates select="." mode="ensure-proper-placement">
      <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
    </xsl:apply-templates>
  </xsl:if>
</xsl:template>

<xsl:template match="idxterm">
  <xsl:value-of select="."/>  <!-- do not pass through phrases -->
</xsl:template>

<!-- [17c] Templates for processing referenced index entries -->

<!-- Process an i1 that has been referenced, either directly through iref or @i1id, or
     indirectly through a referenced i2. If the reference came from an i2 with children,
     or from an i3, $i3val contains the tertiary index value. If the reference came from an i2,
     $i2val contains the secondary entry. If the reference came from iref, these are empty. -->
<xsl:template name="process-i1-entry-by-reference">
  <xsl:param name="i2val"></xsl:param>
  <xsl:param name="i3val"></xsl:param>
  <!-- Create a term (or set) for each idxterm in this i1 -->
  <xsl:for-each select="idxterm">
    <indexterm>
      <xsl:value-of select="."/>
      <xsl:if test="string-length($i2val)>0">     <!-- If there was a referencing i2, create it -->
        <indexterm>
          <xsl:value-of select="$i2val"/>
          <xsl:if test="string-length($i3val)>0"> <!-- If there was a referencing i3, create it -->
            <indexterm><xsl:value-of select="$i3val"/></indexterm>
          </xsl:if>
        </indexterm>
      </xsl:if>
    </indexterm>
  </xsl:for-each>
</xsl:template>

<!-- With referenced terms, the the wrapper is always determined by the element that referenced
     the current term. If one is needed, create it and process the i1. Otherwise, process the i1. -->
<xsl:template match="i1" mode="index-by-reference">
  <xsl:param name="parent-element"/>
  <xsl:param name="wrapper"/>
  <xsl:param name="i2val"></xsl:param>
  <xsl:param name="i3val"></xsl:param>
  <xsl:choose>
    <xsl:when test="($parent-element='refbody' or $parent-element='conbody') and $wrapper='section'">
      <xsl:call-template name="output-required-cleanup-warning"/>
      <section><required-cleanup><xsl:call-template name="process-i1-entry"/></required-cleanup></section>
    </xsl:when>
    <xsl:when test="string-length($wrapper)>0">
      <xsl:element name="{$wrapper}">
        <xsl:call-template name="process-i1-entry-by-reference">
          <xsl:with-param name="i2val"><xsl:value-of select="$i2val"/></xsl:with-param>
          <xsl:with-param name="i3val"><xsl:value-of select="$i3val"/></xsl:with-param>
        </xsl:call-template>
      </xsl:element>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="process-i1-entry-by-reference">
        <xsl:with-param name="i2val"><xsl:value-of select="$i2val"/></xsl:with-param>
        <xsl:with-param name="i3val"><xsl:value-of select="$i3val"/></xsl:with-param>
      </xsl:call-template>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Process an i2 by reference. If it originated with an i3, $i3val contains the value; if it
     came from iref, then $i3val is empty. This is only called when the parent is an i1. -->
<xsl:template name="process-i2-entry-by-reference">
  <xsl:param name="i3val"></xsl:param>
  <xsl:variable name="i2val" select="idxterm"/>
  <!-- For each idxterm in the parent i1, create a term that nests the current i2. If this request
       came from an i3, output that i3 as well. -->
  <xsl:for-each select="parent::i1/idxterm">
    <indexterm>
      <xsl:value-of select="."/>
      <indexterm>
        <xsl:value-of select="$i2val"/>
        <xsl:if test="string-length($i3val)>0">
          <indexterm><xsl:value-of select="$i3val"/></indexterm>
        </xsl:if>
      </indexterm>
    </indexterm>
  </xsl:for-each>
</xsl:template>

<!-- With referenced terms, the the wrapper is always determined by the element that referenced
     the current term. See the xsl:choose for further processing details. -->
<xsl:template match="i2" mode="index-by-reference">
  <xsl:param name="parent-element"/>
  <xsl:param name="wrapper"/>
  <xsl:param name="i3val"></xsl:param> <!-- empty unless this i2 was referenced by an i3 instead of iref -->
  <xsl:choose>
    <!-- When this i2 references an i1, process that i1 with mode index-by-reference. Pass it the
         value of this i2, as well as the necessary wrapper and any i3 value. 
         Because this node was referenced directly, child I3s are ignored. -->
    <xsl:when test="@i1id">
      <xsl:apply-templates select="key('i1list',@i1id)" mode="index-by-reference">
        <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
        <xsl:with-param name="wrapper"><xsl:value-of select="$wrapper"/></xsl:with-param>
        <xsl:with-param name="i2val" select="idxterm"/>
        <xsl:with-param name="i3val" select="$i3val"/>
      </xsl:apply-templates>
    </xsl:when>
    <!-- Otherwise the parent is an i1. If there is a wrapper, create it, and process this referenced i2. -->
    <xsl:when test="string-length($wrapper)>0">
      <xsl:element name="{$wrapper}">
        <xsl:call-template name="process-i2-entry-by-reference">
          <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
          <xsl:with-param name="i3val"><xsl:value-of select="$i3val"/></xsl:with-param>
        </xsl:call-template>
      </xsl:element>
    </xsl:when>
    <!-- The parent is an i1; no wrapper is needed, so process this referenced i2. -->
    <xsl:otherwise>
      <xsl:call-template name="process-i2-entry-by-reference">
        <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
        <xsl:with-param name="i3val"><xsl:value-of select="$i3val"/></xsl:with-param>
      </xsl:call-template>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- This can only occur when the i3 is referenced by an iref element. So, if any wrapper is
     needed, it was created in the iref element. -->
<xsl:template match="i3" mode="index-by-reference">
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <!-- If this references an i2, process that i2 by reference, and pass the value of this i3. -->
    <xsl:when test="@i2id">
      <xsl:apply-templates select="key('i2list',@i2id)" mode="index-by-reference">
        <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
        <xsl:with-param name="i3val" select="idxterm"/>
      </xsl:apply-templates>
    </xsl:when>
    <!-- Otherwise, the parent is an i2; process that i2 by reference, passing the current term. -->
    <xsl:otherwise>
      <xsl:apply-templates select="parent::i2" mode="index-by-reference">
        <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
        <xsl:with-param name="i3val" select="idxterm"/>
      </xsl:apply-templates>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- [17d] Process an iref element. Location is already verified as valid. Create a wrapper if one is
     needed; then call the parse-iref-refids template, which will process each referenced entry. -->
<xsl:template match="iref" mode="ensure-proper-placement">
  <xsl:param name="parent-element"/>
  <xsl:param name="wrapper">
    <xsl:call-template name="find-i1-wrapper">
      <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
    </xsl:call-template>
  </xsl:param>
  <xsl:choose>
    <xsl:when test="($parent-element='refbody' or $parent-element='conbody') and $wrapper='section'">
      <xsl:call-template name="output-required-cleanup-warning"/>
      <section><required-cleanup>
        <xsl:call-template name="parse-iref-refids">
          <xsl:with-param name="refids"><xsl:value-of select="@refids"/></xsl:with-param>
        </xsl:call-template>
      </required-cleanup></section>
    </xsl:when>
    <xsl:when test="string-length($wrapper)>0">
      <xsl:element name="{$wrapper}">
        <xsl:call-template name="parse-iref-refids">
          <xsl:with-param name="refids"><xsl:value-of select="@refids"/></xsl:with-param>
        </xsl:call-template>
      </xsl:element>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="parse-iref-refids">
        <xsl:with-param name="refids"><xsl:value-of select="@refids"/></xsl:with-param>
      </xsl:call-template>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- If the iref is in a valid location, process it. -->
<xsl:template match="iref">
  <xsl:param name="parent-element"/>
  <xsl:variable name="useiref"><xsl:call-template name="inside-valid-index-location"/></xsl:variable>
  <xsl:if test="$useiref='valid'">
    <xsl:apply-templates select="." mode="ensure-proper-placement">
      <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
    </xsl:apply-templates>
  </xsl:if>
</xsl:template>

<!-- Parse the refids attribute on an iref tag. Referenced IDs are separated by spaces.
     Find the first ID, and process it; if there are more, call the template on the remainder. -->
<xsl:template name="parse-iref-refids">
  <xsl:param name="refids"/>
  <xsl:variable name="refid">  <!-- Find the first ID in the refids attribute -->
    <xsl:choose>
      <xsl:when test="contains($refids,' ')"><xsl:value-of select="substring-before($refids,' ')"/></xsl:when>
      <xsl:otherwise><xsl:value-of select="$refids"/></xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:variable name="remainder">
    <xsl:value-of select="substring-after($refids,' ')"/>
  </xsl:variable>

  <!-- Find out what the first ID points to (i1, i2, i3), and apply-templates on that element,
       using the index-by-reference mode. -->
  <xsl:choose>
    <xsl:when test="key('i1list',$refid)">
      <xsl:apply-templates select="key('i1list',$refid)" mode="index-by-reference">
      </xsl:apply-templates>
    </xsl:when>
    <xsl:when test="key('i2list',$refid)">
      <xsl:apply-templates select="key('i2list',$refid)" mode="index-by-reference">
      </xsl:apply-templates>
    </xsl:when>
    <xsl:when test="key('i3list',$refid)">
      <xsl:apply-templates select="key('i3list',$refid)" mode="index-by-reference">
      </xsl:apply-templates>
    </xsl:when>
    <xsl:otherwise> <!-- output-message? invalid reference --> </xsl:otherwise>
  </xsl:choose>

  <!-- Call this template again with any remaining IDs -->
  <xsl:if test="string-length($remainder)>0">
    <xsl:call-template name="parse-iref-refids">
      <xsl:with-param name="refids"><xsl:value-of select="$remainder"/></xsl:with-param>
    </xsl:call-template>
  </xsl:if>
</xsl:template>

<!-- **************************** [18] ModInfo  **************************** -->

<!-- Create a key list for quick look up of definitions -->
<xsl:key name="moditemdefs" match="moditemdef" use="@classname"/>

<!-- Modinfo defaults to a reference topic, unless style=table is specified (next template).
     Each <mod> child will also default to a reference topic. -->
<!-- output-message: conloc to modinfo with style=table won't work -->
<xsl:template match="modinfo">
  <xsl:call-template name="output-divlike">
    <xsl:with-param name="default-type">reference</xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!-- When style=table, create a simple table. The headers for the simpletable are taken
     from the moditemdefs (using the keylist) -->
<xsl:template match="modinfo[@style='table']">
  <xsl:value-of select="$newline"/>
  <xsl:apply-templates select="desc"/>
  <simpletable>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:value-of select="$newline"/>
    <sthead>
      <xsl:for-each select="mod[1]/modname|mod[1]/moditem|mod[1]/moddesc">
        <stentry>
          <!-- use @class to find the title in the key table. Without keys, would need to check
               current modinfo, current divisions, main prolog for both moditemdef and modinfodef/moditemdef -->
          <xsl:apply-templates select="key('moditemdefs',@class)/title"/>
        </stentry>
      </xsl:for-each>
    </sthead>
    <xsl:apply-templates select="*[name()!='desc']"/>
    <xsl:value-of select="$newline"/>
  </simpletable>
</xsl:template>

<xsl:template match="moditemdef/title">
  <xsl:apply-templates/>
</xsl:template>
<xsl:template match="moditemdef/desc"/>

<!-- When creating a table, the desc is a paragraph before the table. Otherwise, it falls into
     the shortdesc element. -->
<xsl:template match="modinfo/desc">
  <xsl:choose>
    <xsl:when test="parent::modinfo[@style='table']">
      <p><xsl:apply-templates/></p>
    </xsl:when>
    <xsl:otherwise>
      <shortdesc><xsl:apply-templates/></shortdesc>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- If this is a table-style MODINFO, this will become a row. Otherwise, create a reference topic. -->
<xsl:template match="mod">
  <xsl:choose>
    <xsl:when test="parent::modinfo[@style='table']">
      <xsl:value-of select="$newline"/>
      <strow><xsl:apply-templates/></strow>
    </xsl:when>
    <xsl:when test="@chunk-filename and contains(system-property('xsl:vendor'), 'SAXON')">
      <saxon:output href="{@chunk-filename}">
        <xsl:call-template name="use-reference-dtd"/>
        <xsl:call-template name="format-mod-without-table"/>
      </saxon:output>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="format-mod-without-table"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Simple template to format a mod as a reference topic. -->
<xsl:template name="format-mod-without-table">
  <reference>
    <xsl:call-template name="add-language-attribute"/>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:value-of select="$newline"/>
    <title>
      <xsl:apply-templates select="key('moditemdefs',modname/@class)/title"/><xsl:text>: </xsl:text>
      <xsl:apply-templates select="modname"/>
    </title>
    <xsl:apply-templates select="moddesc"/>
    <xsl:call-template name="output-topic-prolog"/>
    <xsl:value-of select="$newline"/>
    <refbody>
      <xsl:apply-templates select="moditem"/>
    </refbody>
  </reference>
</xsl:template>

<xsl:template match="modname">
  <xsl:choose>
    <xsl:when test="parent::mod/parent::modinfo[@style='table']">
      <stentry><xsl:apply-templates/></stentry>
    </xsl:when>
    <xsl:otherwise>
      <xsl:apply-templates/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="moddesc">
  <xsl:choose>
    <xsl:when test="parent::mod/parent::modinfo[@style='table']">
      <stentry><xsl:apply-templates/></stentry>
    </xsl:when>
    <xsl:otherwise>
      <shortdesc><xsl:apply-templates/></shortdesc>  <!-- output-message for invalid content? -->
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="moditem">
  <xsl:choose>
    <xsl:when test="parent::mod/parent::modinfo[@style='table']">
      <stentry><xsl:apply-templates/></stentry>
    </xsl:when>
    <xsl:otherwise>
      <section>
        <title><xsl:apply-templates select="key('moditemdefs',@class)/title"/></title>
        <xsl:apply-templates>
          <xsl:with-param name="parent-element">section</xsl:with-param>
        </xsl:apply-templates>
      </section>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- ************************** [19] CITE/BIBLIST ************************** -->


<!-- For CIT elements that reference a bibentry, create the cit and then process that bibentry
     (or libentry, or ibmbibentry, or ibmlibentry). Use the form specified locally. -->
<xsl:template match="cit[@bibid]">
  <xsl:variable name="form">
    <xsl:choose>
      <xsl:when test="@form"><xsl:value-of select="@form"/></xsl:when>
      <xsl:when test="parent::biblist/@form"><xsl:value-of select="parent::biblist/@form"/></xsl:when>
      <xsl:when test="parent::biblist">full</xsl:when>
      <xsl:otherwise>normal</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <cite>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates select="ancestor::*/
                                 *[contains(name(),'prolog')]/
                                 bibentrydefs/*[@id=current()/@bibid]">
      <xsl:with-param name="form" select="$form"/>
    </xsl:apply-templates>
  </cite>
</xsl:template>

<!-- If the cit does not contain a reference, output the contents. Index entries and footnotes must
     be moved out of the cit. Note that this never applies templates on everything, so indices and 
     footnotes will not be misplaced due to fallthrough processing: they are only applied when needed. -->
<xsl:template match="cit">
  <xsl:variable name="form">
    <xsl:choose>
      <xsl:when test="@form"><xsl:value-of select="@form"/></xsl:when>
      <xsl:when test="parent::biblist/@form"><xsl:value-of select="parent::biblist/@form"/></xsl:when>
      <xsl:when test="parent::biblist">full</xsl:when>
      <xsl:otherwise>normal</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:if test="i1|i2|i3|iref">
    <xsl:apply-templates select="i1|i2|i3|iref"/>
  </xsl:if>
  <cite>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates select="bibentry|libentry|ibmbibentry|ibmlibentry">
      <xsl:with-param name="form" select="$form"/>
    </xsl:apply-templates>
  </cite>
  <xsl:if test="fn">
    <xsl:apply-templates select="fn"/>
  </xsl:if>
</xsl:template>

<!-- Output a biblist as an unordered list. Index entries inside of the biblist are moved
     into the first list item; footnotes are moved to the end of the last list item. -->
<xsl:template name="output-standard-biblist">
  <xsl:value-of select="$newline"/>
  <ul>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:for-each select="cit|bibentry|ibmbibentry|libentry|ibmlibentry">
      <xsl:value-of select="$newline"/>
      <li>
        <xsl:if test="position()='1'">
          <xsl:apply-templates select="../i1|../i2|../i3|../iref"/>
        </xsl:if>
        <xsl:apply-templates select="."/>
        <xsl:if test="last()">
          <xsl:apply-templates select="../fn"/>
        </xsl:if>
      </li>
    </xsl:for-each>
    <xsl:value-of select="$newline"/>
  </ul>
</xsl:template>

<!-- Determine what wrapper is needed, if any; output the wrapper, and output the biblist. -->
<xsl:template match="biblist">
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <xsl:when test="$parent-element='refbody'">
      <section><xsl:call-template name="output-standard-biblist"/></section>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="output-standard-biblist"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Not sure yet how to support bibid on the <lq> element, so just map lq to lq. -->
<!-- output-message bibid unsupported? or convert to href? -->
<xsl:template match="lq">
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <xsl:when test="$parent-element='refbody' or
                    $parent-element='conbody' and preceding-sibling::pblk">
      <section><lq>
        <xsl:call-template name="add-default-attributes"/>
        <xsl:apply-templates/>
      </lq></section>
    </xsl:when>
    <xsl:otherwise>
      <lq>
        <xsl:call-template name="add-default-attributes"/>
        <xsl:apply-templates/>
      </lq>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- *********************** [20] BIBENTRY/LIBENTRY ************************ -->

<!-- The mode "resolve-cit" is used frequently in this set of templates. It is used to ensure
     proper fallthrough processing for cit children, especially those (such as title, or docnum)
     which can appear elsewhere with different behavior.
     The output for cit's generally matches PDF behavior, based on the form attribute. The only
     change is for form=full. In PDF, several elements still do not appear in the output when
     form=full. In this transform, all elements will be mapped when form=full. -->

<!-- Find the current form attribute, through inheritance. Note that cit elements will pass this
     as a parameter, in which case this function should not be executed. -->
<xsl:template name="findform">
  <xsl:choose>
    <xsl:when test="parent::cite/@form"><xsl:value-of select="parent::cite/@form"/></xsl:when>
    <xsl:when test="ancestor::biblist/@form"><xsl:value-of select="ancestor::biblist/@form"/></xsl:when>
    <xsl:when test="ancestor::biblist">full</xsl:when>
    <xsl:otherwise>normal</xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="output-title-and-subtitle">
  <xsl:apply-templates select="doctitle/titleblk/title" mode="resolve-cit"/>
  <xsl:apply-templates select="doctitle/titleblk/subtitle" mode="resolve-cit"/>
</xsl:template>

<xsl:template name="output-library-title-and-subtitle">
  <xsl:if test="doctitle/library/titleblk/title">
    <ph>
      <xsl:apply-templates select="doctitle/library/titleblk/title" mode="resolve-cit"/>
      <xsl:apply-templates select="doctitle/library/titleblk/subtitle" mode="resolve-cit"/>
    </ph>
    <xsl:text>: </xsl:text>
  </xsl:if>
</xsl:template>

<!-- When form=full, handle every child in the same order as PDF, plus those PDF does not handle.
     Otherwise, output the same elements that PDF creates.  All elements that are ignored here
     are indicated by comments; they do not make sense in a DITA topic. -->
<xsl:template match="bibentry">
  <xsl:param name="form"><xsl:call-template name="findform"/></xsl:param>
  <xsl:choose>
    <xsl:when test="$form='docnum'">
      <xsl:apply-templates select="ordernum" mode="cit-docnum"/>
    </xsl:when>
    <xsl:when test="$form='title'">
      <xsl:call-template name="output-title-and-subtitle"/>
    </xsl:when>
    <xsl:when test="$form='full'">
      <xsl:call-template name="output-library-title-and-subtitle"/>
      <xsl:call-template name="output-title-and-subtitle"/>
      <xsl:apply-templates select="authors" mode="resolve-cit"/>
      <!-- ignoring desc, externalfilename -->
      <xsl:apply-templates select="filenum" mode="resolve-cit"/>
      <xsl:apply-templates select="ibmdocnum" mode="resolve-cit"/>
      <xsl:apply-templates select="ibmpartnum" mode="resolve-cit"/>
      <xsl:apply-templates select="isbn" mode="resolve-cit"/>
      <xsl:apply-templates select="ordernum" mode="resolve-cit"/>
      <xsl:apply-templates select="prtloc" mode="resolve-cit"/>
      <xsl:apply-templates select="publicid" mode="resolve-cit"/>
      <xsl:apply-templates select="publisher" mode="resolve-cit"/>
    </xsl:when>
    <xsl:otherwise> <!-- when form=normal -->
      <xsl:call-template name="output-library-title-and-subtitle"/>
      <xsl:call-template name="output-title-and-subtitle"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- When form=full, handle every child in the same order as PDF, plus those PDF does not handle.
     Otherwise, output the same elements that PDF creates.  All elements that are ignored here
     are indicated by comments; they do not make sense in a DITA topic. -->
<xsl:template match="ibmbibentry">
  <xsl:param name="form"><xsl:call-template name="findform"/></xsl:param>
  <xsl:choose>
    <xsl:when test="$form='docnum'">
      <xsl:apply-templates select="ibmdocnum" mode="cit-docnum"/>
    </xsl:when>
    <xsl:when test="$form='title'">
      <xsl:call-template name="output-title-and-subtitle"/>
    </xsl:when>
    <xsl:when test="$form='full'">
      <xsl:call-template name="output-library-title-and-subtitle"/>
      <xsl:call-template name="output-title-and-subtitle"/>
      <xsl:apply-templates select="authors" mode="resolve-cit"/>
      <!-- ignoring coverdef, desc, externalfilename -->
      <xsl:apply-templates select="filenum" mode="resolve-cit"/>
      <xsl:apply-templates select="ibmdocnum" mode="resolve-cit"/>
      <xsl:apply-templates select="ibmpartnum" mode="resolve-cit"/>
      <xsl:apply-templates select="isbn" mode="resolve-cit"/>
      <xsl:apply-templates select="ordernum" mode="resolve-cit"/>
      <!-- ignoring origibmdocnum -->
      <xsl:apply-templates select="prtloc" mode="resolve-cit"/>
      <xsl:apply-templates select="publicid" mode="resolve-cit"/>
      <xsl:apply-templates select="publisher" mode="resolve-cit"/>
      <!-- ignoring retkey, volid -->
    </xsl:when>
    <xsl:otherwise> <!-- when form=normal -->
      <xsl:call-template name="output-library-title-and-subtitle"/>
      <xsl:call-template name="output-title-and-subtitle"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- When form=full, handle every child in the same order as PDF, plus those PDF does not handle.
     Otherwise, output the same elements that PDF creates.  All elements that are ignored here
     are indicated by comments; they do not make sense in a DITA topic. -->
<xsl:template match="libentry | ibmlibentry">
  <xsl:param name="form"><xsl:call-template name="findform"/></xsl:param>
  <xsl:choose>
    <xsl:when test="$form='docnum'">
      <!-- bofnum&ordernum are on libentry, others on ibmlibentry -->
      <xsl:apply-templates select="bofnum|ibmbofnum" mode="cit-docnum"/>
      <xsl:apply-templates select="ordernum|ibmpartnum" mode="resolve-cit"/>
    </xsl:when>
    <xsl:when test="$form='title'">
      <xsl:apply-templates select="library/titleblk/title" mode="resolve-cit"/>
      <xsl:apply-templates select="library/titleblk/subtitle" mode="resolve-cit"/>
    </xsl:when>
    <xsl:when test="$form='full'">
      <xsl:apply-templates select="library/titleblk/title" mode="resolve-cit"/>
      <xsl:apply-templates select="library/titleblk/subtitle" mode="resolve-cit"/>
      <xsl:apply-templates select="bofnum|ibmbofnum" mode="resolve-cit"/>
      <xsl:apply-templates select="ordernum|ibmpartnum" mode="resolve-cit"/>
      <!-- ignoring containeddocs, desc, -->
      <xsl:apply-templates select="isbn" mode="resolve-cit"/>
      <xsl:apply-templates select="prtloc" mode="resolve-cit"/>
      <xsl:apply-templates select="publicid" mode="resolve-cit"/>
      <xsl:apply-templates select="publisher" mode="resolve-cit"/>
    </xsl:when>
    <xsl:otherwise> <!-- when form=normal -->
      <xsl:apply-templates select="library/titleblk/title" mode="resolve-cit"/>
      <xsl:apply-templates select="library/titleblk/subtitle" mode="resolve-cit"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Mode cit-docnum is used when form=docnum, with bibentry/ordernum, ibmbibentry/ibmdocnum,
     libentry/bofnum, and ibmlibentry/ibmbofnum. It prevents these from getting the <ph> wrappers
     they get with form=resolve-cit. -->
<xsl:template match="*" mode="cit-docnum">
  <xsl:apply-templates/>
</xsl:template>

<xsl:template match="title" mode="resolve-cit">
  <xsl:apply-templates/>
</xsl:template>

<xsl:template match="authors" mode="resolve-cit">
  <xsl:apply-templates select="author/corp|author/person" mode="resolve-cit"/>
</xsl:template>

<xsl:template match="corp" mode="resolve-cit">
  <xsl:apply-templates select="corpname" mode="resolve-cit"/>
  <xsl:apply-templates select="address" mode="resolve-cit"/>
</xsl:template>

<xsl:template match="person" mode="resolve-cit">
  <xsl:apply-templates select="name" mode="resolve-cit"/>
  <xsl:apply-templates select="address" mode="resolve-cit"/>
</xsl:template>

<xsl:template match="publisher" mode="resolve-cit">
  <xsl:apply-templates select="corpname" mode="resolve-cit"/>
  <xsl:apply-templates select="address" mode="resolve-cit"/>
</xsl:template>

<xsl:template match="address" mode="resolve-cit">
  <xsl:text>, </xsl:text><ph><xsl:apply-templates mode="resolve-cit"/></ph>
</xsl:template>

<xsl:template match="*" mode="resolve-cit">
  <xsl:text>, </xsl:text><ph><xsl:apply-templates/></ph>
</xsl:template>

<xsl:template match="externalfilename"/>
<xsl:template match="externalfilename" mode="resolve-cit"/>

<!-- **************************** [21] OBJLIB ****************************** -->

<!-- The contents of the objlib will be handled by fallthrough processing. The OBJLIB will
     create a file if there is any block or phrase information to be saved. If the objlib
     only contains topics, there is no need for a file to contain them. -->
<xsl:template match="objlib">
  <xsl:choose>
    <!-- If the objlib contains only div-like information, do not create an objlib file.
         <dblk> only contains div-like, and desc is ignored, so also skip these children.
         The objlib should always have @chunk-filename, the check is just in case. -->
    <xsl:when test="@chunk-filename and objlibbody/*[not(@chunk-filename)][not(self::dblk or self::desc)]">
      <saxon:output href="{@chunk-filename}">
        <xsl:call-template name="use-topic-dtd"/>
        <topic>
          <xsl:call-template name="add-language-attribute"/>
          <xsl:call-template name="add-default-attributes"/>
          <xsl:value-of select="$newline"/>
          <title>&nbsp;</title>
          <xsl:apply-templates select="desc"/>
          <xsl:value-of select="$newline"/>
          <body>
            <required-cleanup><xsl:apply-templates/></required-cleanup>
          </body>
        </topic>
      </saxon:output>
    </xsl:when>
    <xsl:otherwise>
      <xsl:apply-templates/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="objlib/desc">
  <shortdesc><xsl:apply-templates/></shortdesc>
</xsl:template>

<xsl:template match="objlibbody">
  <xsl:apply-templates/>
</xsl:template>

<!-- ****************** [22] GENERAL BLOCK-LIKE ELEMENTS ******************* -->

<!-- If in a shortdesc, try to map the paragraph to a phrase. Otherwise, determine if a wrapper is
     needed, and map paragraph to paragraph. -->
<xsl:template match="p">
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <xsl:when test="$parent-element='shortdesc'">
      <ph>
        <xsl:apply-templates/>
      </ph>
    </xsl:when>
    <xsl:when test="$parent-element='refbody' or
                    ($parent-element='conbody' and preceding-sibling::pblk)">
      <xsl:value-of select="$newline"/>
      <section><p>
        <xsl:call-template name="add-default-attributes"/>
        <xsl:apply-templates/>
      </p></section>
    </xsl:when>
    <xsl:otherwise>
      <xsl:value-of select="$newline"/>
      <p>
        <xsl:call-template name="add-default-attributes"/>
        <xsl:apply-templates/>
      </p>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Map pblk to section. -->
<xsl:template name="output-standard-pblk">
  <xsl:value-of select="$newline"/>
  <section>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates>
      <xsl:with-param name="parent-element">section</xsl:with-param>
    </xsl:apply-templates>
  </section>
</xsl:template>

<!-- Pblk can only really map to section, which is only allowed at the <body> level. Anyplace
     else it must be placed in a required-cleanup, which itself may not be in context. We cannot
     just output the contents, because pblk is frequently referenced via conloc, and this would
     result in missing ouput.
     The reason some parent tests use $parent-element and some use parent::* is that the same parent
     may have multiple mappings. For example, if a msgitem maps to a file, the section will be
     allowed; if it maps to section, this section is not allowed. $parent-element makes this check
     simple. However, defn can never hold a pblk, so it is easy to do a parent::* check. -->
<xsl:template match="pblk">
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <xsl:when test="$parent-element='result' or $parent-element='context' or
                    $parent-element='section' or $parent-element='example' or
                    $parent-element='refsyn' or $parent-element='note'">
      <xsl:call-template name="output-required-cleanup-warning"/>
      <required-cleanup><xsl:call-template name="output-standard-pblk"/></required-cleanup>
    </xsl:when>
    <xsl:when test="parent::defn or parent::li">
      <xsl:call-template name="output-required-cleanup-warning"/>
      <required-cleanup><xsl:call-template name="output-standard-pblk"/></required-cleanup>
    </xsl:when>
    <xsl:when test="parent::fig|parent::entry">
      <xsl:call-template name="output-required-cleanup-warning"/>
      <p><required-cleanup><xsl:call-template name="output-standard-pblk"/></required-cleanup></p>
    </xsl:when>
    <xsl:when test="parent::fn">
      <xsl:call-template name="output-required-cleanup-warning"/>
      <ph><required-cleanup><xsl:call-template name="output-standard-pblk"/></required-cleanup></ph>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="output-standard-pblk"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="nitem">
  <xsl:apply-templates/>
</xsl:template>

<!-- As with indices, footnotes can appear in limited locations in DITA, but virtually anywhere in
     IBMIDDoc. In order to make them valid, they frequently need a wrapper; this function determines
     which wrapper is needed. -->
<xsl:template name="find-fn-wrapper">
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <xsl:when test="$parent-element='conbody' and preceding-sibling::pblk">section</xsl:when>
    <xsl:when test="$parent-element='refbody'">section</xsl:when>
    <xsl:when test="parent::title">ph</xsl:when>
    <xsl:when test="parent::dbody">p</xsl:when>
    <xsl:when test="parent::fig">p</xsl:when>
    <xsl:when test="parent::cap">ph</xsl:when>
    <xsl:when test="parent::len">ph</xsl:when>
    <xsl:when test="parent::le">section</xsl:when>
    <xsl:when test="parent::ol or parent::ul or parent::notelist">required-cleanup</xsl:when>
    <xsl:when test="parent::dl or parent::gl or parent::parml">required-cleanup</xsl:when>
    <xsl:when test="parent::dlentry or parent::glentry or parent::parm">required-cleanup</xsl:when>
    <xsl:when test="parent::l">ph</xsl:when>
    <xsl:when test="parent::term">ph</xsl:when>
    <xsl:when test="parent::ph[@style][@style!='base' and @style!='smallcaps']">ph</xsl:when>
    <xsl:when test="parent::code">ph</xsl:when>
    <xsl:when test="parent::msgnum and not(../following-sibling::msgtext)">ph</xsl:when>
    <xsl:when test="parent::msgtext and not(../preceding-sibling::msgnum)">ph</xsl:when>
    <xsl:when test="parent::annot or parent::annotbody">ph</xsl:when>
    <xsl:when test="parent::ednotices|parent::notices|parent::nitem|parent::safety">p</xsl:when>
  </xsl:choose>
</xsl:template>

<!-- This template is only used when we know the <fn> is in a valid location. It will determine
     the proper wrapper, if any; output a warning if the wrapper is required-cleanup; and map
     the fn to fn. -->
<xsl:template match="fn" mode="ensure-proper-placement">
  <xsl:param name="parent-element"/>
  <xsl:variable name="wrapper">
    <xsl:call-template name="find-fn-wrapper">
      <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
    </xsl:call-template>
  </xsl:variable>
  <xsl:if test="$wrapper='required-cleanup'">
    <xsl:call-template name="output-required-cleanup-warning"/>
  </xsl:if>
  <xsl:choose>
    <xsl:when test="string-length($wrapper)>0">
      <xsl:element name="{$wrapper}">
        <fn>
          <xsl:call-template name="add-default-attributes"/>
          <xsl:apply-templates/>
        </fn>
      </xsl:element>
    </xsl:when>
    <xsl:otherwise>
      <fn>
        <xsl:call-template name="add-default-attributes"/>
        <xsl:apply-templates/>
      </fn>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- If a footnote has @refid, treat it the same as an <xref>. The same wrapper will be used that
     is used for a basic footnote.
     E003903: remove @type="fn" from xref. -->
<xsl:template match="fn[@refid]">
  <xsl:variable name="thisfile"><xsl:call-template name="get-current-chunk-name"/></xsl:variable>
  <xsl:variable name="wrapper"><xsl:call-template name="find-fn-wrapper"/></xsl:variable>
  <xsl:if test="$wrapper='required-cleanup'">
    <xsl:call-template name="output-required-cleanup-warning"/>
  </xsl:if>
  <xsl:choose>
    <xsl:when test="parent::frontm or parent::backm">
      <!-- output-message: no equivalent place to put it in DITA -->
    </xsl:when>
    <xsl:when test="string-length($wrapper)>0">
      <xsl:element name="{$wrapper}">
        <xref>
          <xsl:attribute name="href">
            <xsl:apply-templates select="key('xreflist',@refid)[1]" mode="get-xref-target">
              <xsl:with-param name="xreffile"><xsl:value-of select="$thisfile"/></xsl:with-param>
            </xsl:apply-templates>
          </xsl:attribute>
        </xref>
      </xsl:element>
    </xsl:when>
    <xsl:otherwise>
      <xref>
        <xsl:attribute name="href">
          <xsl:apply-templates select="key('xreflist',@refid)[1]" mode="get-xref-target">
            <xsl:with-param name="xreffile"><xsl:value-of select="$thisfile"/></xsl:with-param>
          </xsl:apply-templates>
        </xsl:attribute>
      </xref>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Many footnotes cannot be processed in place, or else they will be out of context in DITA. 
     So, ignore those cases; they will be processed in the appropriate place with the
     ensure-proper-placement mode. -->
<xsl:template match="fn">
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <xsl:when test="parent::dl or parent::dlentry or parent::parml or parent::parm or
                    parent::gl or parent::glentry or parent::ol or parent::ul or parent::notelist or
                    parent::synph or parent::kwd or parent::var or parent::sep or
                    parent::delim or parent::oper or parent::mv or parent::pv or
                    parent::pk or parent::q or parent::term or parent::xph">
      <!-- Will be processed specifically later, not as part of fallthrough -->
    </xsl:when>
    
    <xsl:when test="parent::fig and following-sibling::cap">
      <!-- Will be processed specifically later, not as part of fallthrough -->
    </xsl:when>
    
    <!-- output-message for lers/msglist/frontm/backm: will get lost -->
    <xsl:when test="parent::lers"/>
    <xsl:when test="parent::msglist and $MSGLIST='newfile'"/>
    <xsl:when test="parent::frontm or parent::backm"/>
    
    <xsl:when test="parent::termhd or parent::defnhd">
      <ph><xsl:apply-templates select="." mode="ensure-proper-placement"/></ph>
    </xsl:when>
    <xsl:otherwise>
      <xsl:apply-templates select="." mode="ensure-proper-placement">
        <xsl:with-param name="parent-element"><xsl:value-of select="$parent-element"/></xsl:with-param>
      </xsl:apply-templates>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="fnlist">
  <xsl:call-template name="unsupported-element"/>
</xsl:template>

<xsl:template match="marklist">
  <xsl:call-template name="unsupported-element"/>
</xsl:template>

<xsl:template match="mknote">
  <xsl:call-template name="unsupported-element"/>
</xsl:template>

<!-- This element is no longer valid in IBMIDDoc, so it should always create a warning. The only
     possible mapping is to required-cleanup, although we must be sure that required-cleanup is in
     context. dvcfobj can go anywhere in IDDoc, so it is still possible to get this out of context. -->
<xsl:template match="dvcfobj">
  <xsl:call-template name="output-required-cleanup-warning"/>
  <xsl:choose>
    <xsl:when test="parent::term|parent::fn">
      <ph><required-cleanup><xsl:apply-templates/></required-cleanup></ph>
    </xsl:when>
    <xsl:when test="parent::fig">
      <p><required-cleanup><xsl:apply-templates/></required-cleanup></p>
    </xsl:when>
    <xsl:otherwise>
      <required-cleanup><xsl:apply-templates/></required-cleanup>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Map annot to draft-comment, after any wrapper has been created. -->
<xsl:template name="output-standard-annot">
  <draft-comment>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates/>
  </draft-comment>
</xsl:template>

<!-- Create a wrapper if needed, and map the annot. -->
<xsl:template match="annot">
  <xsl:choose>
    <xsl:when test="parent::dbody|parent::fig">
      <p><xsl:call-template name="output-standard-annot"/></p>
    </xsl:when>
    <xsl:when test="parent::fn">
      <ph><xsl:call-template name="output-standard-annot"/></ph>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="output-standard-annot"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="annotbody">
  <xsl:apply-templates/>
</xsl:template>

<!-- Address can appear in numerous locations, with different behavior based on location.
     This template calls out every possible parent of address, even those (such as person)
     which should never get here. Each parent is given the best equivalent mapping. -->
<xsl:template match="address">
  <xsl:choose>
    <xsl:when test="parent::annotbody or parent::attention or parent::caution or
                    parent::danger or parent::defn or parent::desc or parent::entry or
                    parent::fn or parent::li or parent::lq or parent::notebody or
                    parent::p or parent::warning">
      <lines><xsl:apply-templates/></lines>
    </xsl:when>
    <xsl:when test="parent::l or parent::ph or parent::q or parent::synnote">
      <!-- output-message -->
      <ph><xsl:apply-templates/></ph>
    </xsl:when>
    <xsl:when test="parent::lines">
      <!-- output-message? -->
      <ph><xsl:apply-templates/></ph>
    </xsl:when>
    <xsl:when test="parent::md or parent::mknote">
      <!-- output-message? -->
      <ph><xsl:apply-templates/></ph>
    </xsl:when>
    <xsl:when test="parent::bridge or parent::compcmt or parent::moddesc or parent::moditem">
      <lines><xsl:apply-templates/></lines>
    </xsl:when>
    <xsl:when test="parent::person or parent::corp or parent::publisher">
      <!-- should be handled with meta-data, so this should not get called -->
      <xsl:apply-templates/>
    </xsl:when>
  </xsl:choose>
</xsl:template>

<!-- Everything in address should come out as a phrase. We might want to save the element
     name on the outputclass attribute? -->
<xsl:template match="ibmmail|internet|phone|postalcode|vnet|webpage">
  <ph><xsl:apply-templates/></ph>
</xsl:template>

<!-- BRIDGE: these elements can rarely be preserved inline. 
     E003814: Instead of pulling bridge to the front of a list, create
     an element for it inline, and continue to place it in required-cleanup. -->
<xsl:template match="ol/bridge|ul/bridge|notelist/bridge|liblk/bridge" priority="1">
  <xsl:call-template name="output-required-cleanup-warning"/>
  <li><required-cleanup><xsl:apply-templates/></required-cleanup></li>
</xsl:template>
<xsl:template match="ol/bridge|liblk/bridge" mode="output-as-steps">
  <xsl:call-template name="output-required-cleanup-warning"/>
  <step><cmd><required-cleanup><xsl:apply-templates/></required-cleanup></cmd></step>
</xsl:template>

<xsl:template match="dl/bridge|dlblk/bridge|gl/bridge|glblk/bridge" priority="1">
  <xsl:call-template name="output-required-cleanup-warning"/>
  <dlentry><dt/><dd><required-cleanup><xsl:apply-templates/></required-cleanup></dd></dlentry>
</xsl:template>

<xsl:template match="parml/bridge|parmblk/bridge" priority="1">
  <xsl:call-template name="output-required-cleanup-warning"/>
  <plentry><pt/><pd><required-cleanup><xsl:apply-templates/></required-cleanup></pd></plentry>
</xsl:template>

<!-- The bridge in other locations are given a best-possible mapping, although there
     is no good DITA equivalent. -->
<xsl:template match="bridge">
  <xsl:param name="parent-element"/>
  <xsl:choose>
    <xsl:when test="$parent-element='refbody'">
      <section><xsl:apply-templates/></section>
    </xsl:when>
    <xsl:when test="$parent-element='conbody' and preceding-sibling::pblk">
      <section><xsl:apply-templates/></section>
    </xsl:when>
    <xsl:when test="parent::dbody|parent::fig">
      <p><xsl:apply-templates/></p>
    </xsl:when>
    <xsl:otherwise>
      <xsl:apply-templates/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- ********************** [23] GENERAL PHRASE-LIKE ELEMENTS ********************** -->

<!-- This template is called to map a phrase with @style to equivelent DITA style tags. -->
<!-- P017334: also works for @style-name, which is always upper case -->
<xsl:template name="create-phrase-with-formatting">
  <xsl:param name="ph-style"/>
  <xsl:choose>
    <xsl:when test="$ph-style='base' or $ph-style='BASE'">
      <ph><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></ph>
    </xsl:when>
    <xsl:when test="$ph-style='bold' or $ph-style='BOLD'">
      <b><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></b>
    </xsl:when>
    <xsl:when test="$ph-style='italic' or $ph-style='ITALIC'">
      <xsl:choose>
        <xsl:when test="../@class='ibmfilepath' or ../@class='filepath'">
          <varname><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></varname>
        </xsl:when>
        <xsl:otherwise>
          <i><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></i>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>
    <xsl:when test="$ph-style='underlined' or $ph-style='UNDERLINED'">
      <u><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></u>
    </xsl:when>
    <xsl:when test="$ph-style='monospaced' or $ph-style='MONOSPACED'">
      <tt><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></tt>
    </xsl:when>
    <xsl:when test="$ph-style='subscript' or $ph-style='SUBSCRIPT'">
      <sub><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></sub>
    </xsl:when>
    <xsl:when test="$ph-style='superscript' or $ph-style='SUPERSCRIPT'">
      <sup><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></sup>
    </xsl:when>
    <xsl:when test="$ph-style='bold italic' or $ph-style='BOLD ITALIC'">
      <b><xsl:call-template name="add-default-attributes"/><i><xsl:apply-templates/></i></b>
    </xsl:when>
    <xsl:when test="$ph-style='underlined bold' or $ph-style='UNDERLINED BOLD'">
      <u><xsl:call-template name="add-default-attributes"/><b><xsl:apply-templates/></b></u>
    </xsl:when>
    <xsl:when test="$ph-style='underlined italic' or $ph-style='UNDERLINED ITALIC'">
      <u><xsl:call-template name="add-default-attributes"/><i><xsl:apply-templates/></i></u>
    </xsl:when>
    <xsl:when test="$ph-style='underlined bold italic' or $ph-style='UNDERLINED BOLD ITALIC'">
      <u><xsl:call-template name="add-default-attributes"/><b><i><xsl:apply-templates/></i></b></u>
    </xsl:when>
    <xsl:when test="$ph-style='underlined smallcaps' or $ph-style='UNDERLINED SMALLCAPS'">
      <!-- output-message for smallcaps? -->
      <ph><xsl:call-template name="add-default-attributes"/><u><xsl:apply-templates/></u></ph>
    </xsl:when>
    <xsl:when test="$ph-style='smallcaps' or $ph-style='SMALLCAPS'">
      <!-- output-message for smallcaps? -->
      <ph><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></ph>
    </xsl:when>
    <xsl:otherwise>
      <ph><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></ph>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- If a phrase contains a class attribute that matches one of our new phrase-based elements,
     them map the phrase to that element. Otherwise, if @style is specified, use that to
     create the proper formatting. If @class and not @style, but @class is not a know value,
     try to take the style from the classdef element. The final fallback is to <ph>, with an
     attempt to save the class value. -->
<xsl:template match="ph">
  <xsl:choose>
    <xsl:when test="../@class='ibmcommand' or ../@class='cmdname' or
                    ../@class='apiname' or ../@class='option' or
                    ../@class='parmname' or ../@class='varname' or
                    ../@class='wintitle'">  
      <!-- P017677: the parents do not allow children -->
      <xsl:apply-templates/>
    </xsl:when>
    <xsl:when test="@class and key('classdef',@class)">
      <xsl:choose>
        <!-- Treatment of @class on phrase based on issue E002932 -->
        <xsl:when test="@class='ibmguicontrol' or @class='uicontrol'">
          <uicontrol><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></uicontrol>
        </xsl:when>
        <xsl:when test="@class='ibmfilepath' or @class='filepath'">
          <filepath><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></filepath>
        </xsl:when>
        <xsl:when test="@class='ibmcommand' or @class='cmdname'">
          <cmdname><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></cmdname>
        </xsl:when>
        <xsl:when test="@class='apiname'">
          <apiname><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></apiname>
        </xsl:when>
        <xsl:when test="@class='option'">
          <option><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></option>
        </xsl:when>
        <xsl:when test="@class='parmname'">
          <parmname><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></parmname>
        </xsl:when>
        <xsl:when test="@class='systemoutput'">
          <systemoutput><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></systemoutput>
        </xsl:when>
        <xsl:when test="@class='userinput'">
          <userinput><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></userinput>
        </xsl:when>
        <xsl:when test="@class='varname'">
          <varname><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></varname>
        </xsl:when>
        <xsl:when test="@class='wintitle'">
          <wintitle><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></wintitle>
        </xsl:when>
        <xsl:when test="@style">
          <xsl:call-template name="create-phrase-with-formatting">
            <xsl:with-param name="ph-style"><xsl:value-of select="@style"/></xsl:with-param>
          </xsl:call-template>
        </xsl:when>
        <xsl:when test="key('classdef',@class)/@style">
          <xsl:call-template name="create-phrase-with-formatting">
            <xsl:with-param name="ph-style"><xsl:value-of select="key('classdef',@class)/@style"/></xsl:with-param>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <ph>
            <!-- If an actual outputclass is referenced, this default one will be overridden -->
            <xsl:attribute name="outputclass"><xsl:value-of select="@class"/></xsl:attribute>
            <xsl:call-template name="add-default-attributes"/>
            <xsl:apply-templates/>
          </ph>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>
    <xsl:when test="@style">
      <xsl:call-template name="create-phrase-with-formatting">
        <xsl:with-param name="ph-style"><xsl:value-of select="@style"/></xsl:with-param>
      </xsl:call-template>
    </xsl:when>
    <!-- P017334: when using /publish, @style comes through as @style-name -->
    <xsl:when test="@style-name">
      <xsl:call-template name="create-phrase-with-formatting">
        <xsl:with-param name="ph-style"><xsl:value-of select="@style-name"/></xsl:with-param>
      </xsl:call-template>
    </xsl:when>
    <!-- If it uses @conloc, and the target has a style, make sure this element creates that style -->
    <xsl:when test="@conloc and (key('xreflist',@conloc)[1]/@class | key('xreflist',@conloc)[1]/@style)">
      <xsl:choose>
        <xsl:when test="key('xreflist',@conloc)[1]/@class='ibmguicontrol' or 
                        key('xreflist',@conloc)[1]/@class='uicontrol'">
          <uicontrol><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></uicontrol>
        </xsl:when>
        <xsl:when test="key('xreflist',@conloc)[1]/@class='ibmfilepath' or 
                        key('xreflist',@conloc)[1]/@class='filepath'">
          <filepath><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></filepath>
        </xsl:when>
        <xsl:when test="key('xreflist',@conloc)[1]/@class='ibmcommand' or 
                        key('xreflist',@conloc)[1]/@class='cmdname'">
          <cmdname><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></cmdname>
        </xsl:when>
        <xsl:when test="key('xreflist',@conloc)[1]/@class='apiname'">
          <apiname><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></apiname>
        </xsl:when>
        <xsl:when test="key('xreflist',@conloc)[1]/@class='option'">
          <option><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></option>
        </xsl:when>
        <xsl:when test="key('xreflist',@conloc)[1]/@class='parmname'">
          <parmname><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></parmname>
        </xsl:when>
        <xsl:when test="key('xreflist',@conloc)[1]/@class='systemoutput'">
          <systemoutput><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></systemoutput>
        </xsl:when>
        <xsl:when test="key('xreflist',@conloc)[1]/@class='userinput'">
          <userinput><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></userinput>
        </xsl:when>
        <xsl:when test="key('xreflist',@conloc)[1]/@class='varname'">
          <varname><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></varname>
        </xsl:when>
        <xsl:when test="key('xreflist',@conloc)[1]/@class='wintitle'">
          <wintitle><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></wintitle>
        </xsl:when>
        <xsl:when test="key('xreflist',@conloc)[1]/@style">
          <xsl:call-template name="create-phrase-with-formatting">
            <xsl:with-param name="ph-style"><xsl:value-of select="key('xreflist',@conloc)[1]/@style"/></xsl:with-param>
          </xsl:call-template>
        </xsl:when>
        <xsl:when test="key('classdef',key('xreflist',@conloc)[1]/@class)/@style">
          <xsl:call-template name="create-phrase-with-formatting">
            <xsl:with-param name="ph-style"><xsl:value-of select="key('classdef',key('xreflist',@conloc)[1]/@class)/@style"/></xsl:with-param>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise><ph><xsl:call-template name="add-default-attributes"/></ph></xsl:otherwise>
      </xsl:choose>
    </xsl:when>
    <xsl:otherwise>
      <ph><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></ph>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="apl">
  <codeph>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates/>
  </codeph>
</xsl:template>

<xsl:template match="bin">
  <ph><xsl:text>B'</xsl:text><xsl:apply-templates/><xsl:text>'</xsl:text></ph>
</xsl:template>

<xsl:template match="char">
  <ph><xsl:text>"</xsl:text><xsl:apply-templates/><xsl:text>"</xsl:text></ph>
</xsl:template>

<xsl:template match="date">
  <ph><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></ph>
</xsl:template>

<xsl:template match="dec | num">
  <ph><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></ph>
</xsl:template>

<xsl:template match="formula">
  <!-- output-message: not supportable outside of bookmaster. -->
  <xsl:choose>
    <xsl:when test="parent::fig">
      <p><codeph><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></codeph></p>
    </xsl:when>
    <xsl:otherwise>
      <codeph><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></codeph>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="hex">
  <ph><xsl:text>X'</xsl:text><xsl:apply-templates/><xsl:text>'</xsl:text></ph>
</xsl:template>

<xsl:template match="md">
  <ph><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></ph>
</xsl:template>

<xsl:template match="mv">
  <xsl:apply-templates select="i1|i2|i3|iref" mode="ensure-proper-placement"/>
  <varname>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:apply-templates/>
  </varname>
  <xsl:apply-templates select="fn" mode="ensure-proper-placement"/>
</xsl:template>

<xsl:template match="oct">
  <ph><xsl:text>O'</xsl:text><xsl:apply-templates/><xsl:text>'</xsl:text></ph>
</xsl:template>

<xsl:template match="pk">
  <xsl:apply-templates select="i1|i2|i3|iref" mode="ensure-proper-placement"/>
  <synph>
    <xsl:call-template name="add-default-attributes"/>
    <kwd>
      <xsl:apply-templates select="@optreq"/>
      <xsl:apply-templates/>
    </kwd>
  </synph>
  <xsl:apply-templates select="fn" mode="ensure-proper-placement"/>
</xsl:template>

<xsl:template match="pv">
  <xsl:apply-templates select="i1|i2|i3|iref" mode="ensure-proper-placement"/>
  <synph>
    <xsl:call-template name="add-default-attributes"/>
    <var>
      <xsl:apply-templates select="@optreq"/>
      <xsl:apply-templates/>
    </var>
  </synph>
  <xsl:apply-templates select="fn" mode="ensure-proper-placement"/>
</xsl:template>

<xsl:template match="q">
  <!-- output-message for bibid? -->
  <xsl:apply-templates select="i1|i2|i3|iref" mode="ensure-proper-placement"/>
  <q><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></q>
  <xsl:apply-templates select="fn" mode="ensure-proper-placement"/>
</xsl:template>

<xsl:template match="refkey">
  <xsl:choose>
    <xsl:when test="parent::fig">
      <p><ph><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></ph></p>
    </xsl:when>
    <xsl:otherwise>
      <ph><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></ph>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- When processing div-like elements, retkey is usually pulled out and processed as needed. So, we can
     just output the contents. However, when LERS is the parent, there is no good place to output it. -->
<xsl:template match="retkey">
  <xsl:choose>
    <xsl:when test="parent::lers">
      <!-- output-message -->
    </xsl:when>
    <xsl:otherwise>
      <xsl:apply-templates/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="term">
  <xsl:apply-templates select="i1|i2|i3|iref" mode="ensure-proper-placement"/>
  <term><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></term>
  <xsl:apply-templates select="fn" mode="ensure-proper-placement"/>
</xsl:template>

<!-- Map trademark to trademark. The first 3 attributes are all required in IDDoc, so they
     will always be present. @tmowner is optional. -->
<xsl:template match="tm">
  <tm>
    <xsl:call-template name="add-default-attributes"/>
    <xsl:attribute name="trademark"><xsl:value-of select="@trademark"/></xsl:attribute>
    <xsl:attribute name="tmtype"><xsl:value-of select="@tmtype"/></xsl:attribute>
    <xsl:attribute name="tmclass"><xsl:value-of select="@tmclass"/></xsl:attribute>
    <xsl:apply-templates select="@tmowner"/>
    <xsl:apply-templates/>
  </tm>
</xsl:template>

<xsl:template match="tm/@tmowner">
  <xsl:attribute name="tmowner"><xsl:value-of select="."/></xsl:attribute>
</xsl:template>

<xsl:template match="xph">
  <xsl:apply-templates select="i1|i2|i3|iref" mode="ensure-proper-placement"/>
  <codeph><xsl:call-template name="add-default-attributes"/><xsl:apply-templates/></codeph>
  <xsl:apply-templates select="fn" mode="ensure-proper-placement"/>
</xsl:template>

<!-- ******************** [23] Smalldoc (Flyer Format) ***************** -->
<xsl:template match="smalldoc">
  <xsl:call-template name="output-divlike"/>
  <xsl:apply-templates select="smalldocprolog/ednotices"/>
</xsl:template>

<!-- Prolog information is processed directly (only contains the maspart, and ednotices) -->
<xsl:template match="smalldocprolog"/>

<xsl:template match="masthead">
  <title>
    <xsl:apply-templates select="mastpart"/>
  </title>
</xsl:template>

<xsl:template match="mastpart">
  <ph>
    <xsl:choose>
      <xsl:when test="@content='company' and /ibmiddoc/@company"><xsl:value-of select="/ibmiddoc/@company"/></xsl:when>
      <xsl:when test="@content='company'">IBM</xsl:when>
      <!-- The brand values should already be resolved in the XID file, so "logo" is translated; just
           in case something goes wrong, have support here for English only. -->
      <xsl:when test="@brand and /ibmiddoc/@brand='eserver-white'">e(logo)server</xsl:when>
      <xsl:when test="@brand and /ibmiddoc/@brand='eserver-black'">e(logo)server</xsl:when>
      <xsl:when test="@brand"/>
      <xsl:when test="@content='none'"/>
      <xsl:otherwise><xsl:value-of select="."/></xsl:otherwise>
    </xsl:choose>
  </ph>
  <xsl:if test="following-sibling::mastpart">&nbsp;&nbsp;</xsl:if>
</xsl:template>


<!-- ******************** [25] Recognized Processing Instructions ***************** -->

<!-- Currently, the only recognized PI is html-comment, which is formatted in iddoc as <?html:comment text... > -->
<xsl:template match="processing-instruction()">
  <xsl:if test="name()='html-comment'">
    <xsl:processing-instruction name="dita-comment"><xsl:value-of select="."/></xsl:processing-instruction>
  </xsl:if>
</xsl:template>

<!-- ***************************************************************** -->

<xsl:template match="*">
  <xsl:apply-templates/>
</xsl:template>

</xsl:stylesheet>
