<?xml version="1.0"?>
<!--  
      Take in an HTML Help contents file and convert it to a DITA map.
      Allow an optional INDEX parameter to specify an associated HTML Help
      index file; all terms associated with a topic will be added to the
      first occurrence of that topic in the DITA map.
-->

<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                version="1.0">

<xsl:include href="output-message.xsl"/>

<xsl:output method="xml"
            encoding="utf-8"
            doctype-system="..\dtd\map.dtd"
            doctype-public="-//IBM//DTD DITA Map//EN"
/>

<!-- Set the prefix for error message numbers -->
<xsl:variable name="msgprefix">IDXS</xsl:variable>
<xsl:variable name="newline"><xsl:text>
</xsl:text></xsl:variable>

<!-- Key list to store the topic names. Make sure index entries are only
     added to a topic the first time it appears. -->
<xsl:key name="hrefs" match="param[@name='Local']" use="@value"/>

<!-- *************************** Command line parameters *********************** -->

<!-- Take in the HEML Help index file name -->
<xsl:param name="INDEX"/>

<!-- $files starts out with a list of every href that any index entry points to.
     Go through the list. If any target topic is NOT part of the HHC file, issue
     a warning that the index entry will be lost. -->
<xsl:template name="verifyIndex">
  <xsl:param name="files"/>
  <xsl:param name="checked"/> <!-- Files that have already been checked (only issue message once per file) -->
  <xsl:choose>
    <!-- // // appears before and after each href. If the variable starts with this,
         call the template again with whatever comes after. -->
    <xsl:when test="starts-with($files,'// //')">
      <xsl:call-template name="verifyIndex">
        <xsl:with-param name="files" select="substring-after($files,'// //')"/>
        <xsl:with-param name="checked" select="concat($checked,'// //')"/>
      </xsl:call-template>
    </xsl:when>
    <!-- If there are any terms left, find the first one, and issue a message if needed. 
         Then call the template again with remaining values, and add this one to checked values. -->
    <xsl:when test="contains($files,'// //')">
      <xsl:variable name="onefile">
        <xsl:choose>
          <!-- If the file reference has #, after the extension, drop the anchor ID -->
          <xsl:when test="contains(substring-before($files,'// //'),'#') and
                          not(contains(substring-after(substring-before($files,'// //'),'#'),'.'))">
            <xsl:value-of select="substring-before(substring-before($files,'// //'),'#')"/>
          </xsl:when>
          <xsl:otherwise><xsl:value-of select="substring-before($files,'// //')"/></xsl:otherwise>
        </xsl:choose>
      </xsl:variable>
        
      <xsl:if test="not(key('hrefs',$onefile)) and not(contains($checked,concat('// //',$onefile,'// //')))">
        <xsl:call-template name="output-message">
          <xsl:with-param name="msg">One or more index entries in your project reference <xsl:value-of select="$onefile"/>,
but that topic is not part of your contents. The index entry will not be
added to your map.</xsl:with-param>
          <xsl:with-param name="msgnum">055</xsl:with-param>
          <xsl:with-param name="msgsev">W</xsl:with-param>
        </xsl:call-template>
      </xsl:if>
      <xsl:call-template name="verifyIndex">
        <xsl:with-param name="files" select="substring-after($files,'// //')"/>
        <xsl:with-param name="checked" select="concat($checked,$onefile,'// //')"/>
      </xsl:call-template>
    </xsl:when>
  </xsl:choose>
</xsl:template>

<!-- *********************************************************************************
     Setup the HTML wrapper for the table of contents.
     ********************************************************************************* -->
<xsl:template match="/">
  <xsl:if test="$INDEX">
    <xsl:variable name="indexList">
      <xsl:for-each select="document($INDEX, /)//param[@name='Local']">
        <xsl:text/>// //<xsl:value-of select="@value"/>// //<xsl:text/>
      </xsl:for-each>
    </xsl:variable>
    <xsl:call-template name="verifyIndex">
      <xsl:with-param name="files" select="$indexList"/>
    </xsl:call-template>
  </xsl:if>
  <xsl:apply-templates/>
</xsl:template>

<xsl:template match="html">
  <map>
    <xsl:apply-templates select="body/*"/>
  </map>
</xsl:template>

<!-- Null out any object tags that fall through (none should). -->
<xsl:template match="object"/>

<!-- Start the main TOC, or a new branch inside the TOC -->
<xsl:template match="ul">
  <xsl:apply-templates/>
</xsl:template>

<!-- Match a TOC entry -->
<xsl:template match="li">
  <xsl:choose>
    <!-- If there is a file name, create a topicref. Apply-templates on the params
         to pick up a navtitle and the href. -->
    <xsl:when test="object/param[@name='Local']">
      <topicref>
        <xsl:apply-templates select="object/param"/>
        <!-- If this is the first time this HREF appears, check the index for new terms. -->
        <xsl:if test="generate-id(object/param[@name='Local'])=generate-id((key('hrefs',object/param[@name='Local']/@value))[1])">
          <xsl:if test="string-length($INDEX)>0">
            <xsl:apply-templates select="document($INDEX, /)/html/body/ul[1]" mode="primaryIndex">
              <xsl:with-param name="fileName"><xsl:value-of select="object/param[@name='Local']/@value"/></xsl:with-param>
            </xsl:apply-templates>
          </xsl:if>
        </xsl:if>
        <!-- Process any TOC branches from this point -->
        <xsl:apply-templates select="ul"/>
      </topicref>
    </xsl:when>
    <!-- If there is a name but no HREF, create a topichead, and process sub-trees. -->
    <xsl:when test="object/param[@name='Name']">
      <topichead>
        <xsl:apply-templates select="object/param"/>
        <xsl:apply-templates select="ul"/>
      </topichead>
    </xsl:when>
    <!-- Otherwise not sure what this LI was for; create a topicgroup for grouping -->
    <xsl:otherwise>
      <topicgroup><xsl:apply-templates select="ul"/></topicgroup>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Match param within an LI in the TOC. If it has a Name, add the navtitle. If
     it has a Local value, turn this in to href. Drop the htm or html extension,
     and convert it to dita -->
<xsl:template match="param">
  <xsl:choose>
    <xsl:when test="@name='Name'">
      <xsl:attribute name="navtitle"><xsl:value-of select="@value"/></xsl:attribute>
    </xsl:when>
    <xsl:when test="@name='Local'">
      <xsl:attribute name="href">
      <xsl:choose>
        <xsl:when test="contains(@value,'.htm')">
          <xsl:value-of select="substring-before(@value,'.htm')"/>.dita<xsl:text/>
        </xsl:when>
        <xsl:when test="contains(@value,'.HTM')">
          <xsl:value-of select="substring-before(@value,'.HTM')"/>.dita<xsl:text/>
        </xsl:when>
        <xsl:otherwise><xsl:value-of select="@value"/></xsl:otherwise>
      </xsl:choose>
      </xsl:attribute>
      <!-- If the extension was converted to DITA, ignore this. Otherwise, set
           the format attribute based on http, ftp, or the file extension. -->
      <xsl:choose>
        <xsl:when test="contains(@value,'.htm') or contains(@value,'.HTM')"/>
        <xsl:when test="starts-with(@value,'HTTP:') or starts-with(@value,'http:') or
                        starts-with(@value,'HTTPS:') or starts-with(@value,'https:') or
                        starts-with(@value,'ftp:') or starts-with(@value,'FTP:')">
          <xsl:attribute name="format">html</xsl:attribute>
        </xsl:when>
        <xsl:when test="contains(@value,'.')">
          <xsl:attribute name="format"><xsl:value-of select="substring-after(@value,'.')"/></xsl:attribute>
        </xsl:when>
      </xsl:choose>
    </xsl:when>
  </xsl:choose>
</xsl:template>

<!-- Match the main ul inside the index file. If there are any entries pointing
     to the current file, create the topicmeta wrapper, and process the entries. -->
<xsl:template match="body/ul" mode="primaryIndex">
  <xsl:param name="fileName"/>
  <xsl:if test="descendant::param[@name='Local'][starts-with(@value,$fileName)]">
    <topicmeta><keywords>
      <xsl:apply-templates select="li[descendant::param[@name='Local'][starts-with(@value,$fileName)]]" mode="nestedIndex">
        <xsl:with-param name="fileName"><xsl:value-of select="$fileName"/></xsl:with-param>
      </xsl:apply-templates>
    </keywords></topicmeta>
  </xsl:if>
</xsl:template>

<!-- Match a list item. If apply-templates matched this LI, then we know it
     or a sub-term should be in the current file. -->
<xsl:template match="li" mode="nestedIndex">
  <xsl:param name="fileName"/>   <!-- Current file getting processed -->
  <xsl:param name="nestedTerm"/> <!-- If there was a second term on a parent term, keep it -->
  <xsl:variable name="mainTerm"> <!-- The main term for this LI is the first param with name='Name' -->
    <xsl:value-of select="object/param[@name='Name'][1]/@value"/>
  </xsl:variable>
  <!-- If there are two Names in this item: the first is the main term, and the
       second is specific to this file. The Name for this file is one which
       lies immediately before the param pointing to the file. -->
  <xsl:variable name="secondaryTerm">
    <xsl:if test="object/param[@name='Name'][2]">
      <xsl:if test="object/param[@name='Local'][starts-with(@value,$fileName)]">
        <xsl:value-of select="object/param[@name='Local'][starts-with(@value,$fileName)][1]/
                                     preceding-sibling::param[@name='Name'][1]/@value"/>
      </xsl:if>
    </xsl:if>
  </xsl:variable>
  <indexterm>
    <xsl:value-of select="$mainTerm"/>
    <xsl:choose>
      <xsl:when test="ul/descendant::param[@name='Local'][starts-with(@value,$fileName)]">
        <xsl:apply-templates select="ul/li[descendant::param[@name='Local'][starts-with(@value,$fileName)]]" mode="nestedIndex">
          <xsl:with-param name="fileName"><xsl:value-of select="$fileName"/></xsl:with-param>
          <xsl:with-param name="nestedTerm">
            <xsl:choose>
              <xsl:when test="string-length($secondaryTerm)>0"><xsl:value-of select="$secondaryTerm"/></xsl:when>
              <xsl:when test="string-length($nestedTerm)>0"><xsl:value-of select="$nestedTerm"/></xsl:when>
            </xsl:choose>
          </xsl:with-param>
        </xsl:apply-templates>
      </xsl:when>
      <xsl:when test="string-length($secondaryTerm)>0 and normalize-space($secondaryTerm)!=normalize-space($mainTerm)">
        <indexterm><xsl:value-of select="$secondaryTerm"/></indexterm>
      </xsl:when>
      <xsl:when test="string-length($nestedTerm)>0 and normalize-space($nestedTerm)!=normalize-space($mainTerm)">
        <indexterm><xsl:value-of select="$nestedTerm"/></indexterm>
      </xsl:when>
    </xsl:choose>
  </indexterm>
</xsl:template>

</xsl:stylesheet>
