<?xml version="1.0" encoding="utf-8" ?>
<!--(c) Copyright IBM Corp. (This work is unpublished.)-->

<!-- 20030225: RDA   When an element uses @conref, local attributes should be saved.
                     Added conref-target and resolve-attribute modes.
     20031231: RDA   When conref pulls in an item that references an image, the path
                     to that image should remain valid
     20040229: RDA   If a conref pulls in an xref, the href may also need adjusting.
                     If the href is external, leave it alone. If it is to the current
                     file (begins with #), a file name must be added. 
     20040331: RDA   Warn if topics contain duplicate IDs after running conref. May
                     happen when pulling in a block that has sub-elements with IDs. 
     20040727: RDA   Updated for better domains checking. If the domains value on a
                     target's topic is equal to or a subset of the domains on the
                     referencing topic, the reference is valid. Current code only
                     checks for a match. Superset on the target is invalid, because
                     it may contain additional, unknown domains elements. -->

<xsl:stylesheet version="1.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:exsl="http://exslt.org/common">

<xsl:include href="output-message.xsl"/>

<!-- Define the error message prefix identifier -->
<xsl:variable name="msgprefix">IDXS</xsl:variable>

<xsl:param name="WORKDIR" select="'./'"/>
<xsl:param name="PROJDIR" select="'.'"/>
<xsl:param name="DBG" select="no"/>

<!-- P017453: Place the entire result tree in a variable, and then output the
     variable. While sending to output, check for duplicated IDs. -->
<xsl:template match="/">
  <!-- This works for Saxon. Other XSLT parsers may need the node-set() function
       in order to process the variable (usually available with the exsl namespace). -->
  <xsl:variable name="resolved-tree">
    <xsl:apply-templates/>
  </xsl:variable>
  <xsl:apply-templates select="$resolved-tree" mode="verify-and-output"/>
</xsl:template>

<!-- Take in a string of ID values, separated by spaces. Search for the first
     one in the remainder of the string. If it appears more than once, the
     ID is a duplicate, so put out a message. Call the template again with
     the remaining ID values. -->
<xsl:template name="checkDuplicateIds">
  <xsl:param name="idValues"/>
  <!-- If idValues contains a space, there are at least 2 IDs. -->
  <xsl:if test="contains($idValues,' ')">
    <xsl:variable name="oneValue">   <!-- The first ID value. -->
      <xsl:text> </xsl:text><xsl:value-of select="substring-before($idValues,' ')"/><xsl:text> </xsl:text>
    </xsl:variable>
    <xsl:variable name="compareTo">  <!-- All IDs except the first value. -->
      <xsl:text> </xsl:text><xsl:value-of select="substring-after($idValues,' ')"/><xsl:text> </xsl:text>
    </xsl:variable>
    <xsl:if test="contains($compareTo,$oneValue)">
      <xsl:call-template name="output-message">
        <xsl:with-param name="msg">After resolving conref values, the <xsl:value-of select="name()"/> topic with id="<xsl:value-of select="@id"/>"
contains multiple elements with id="<xsl:value-of select="normalize-space($oneValue)"/>". If this is the first
message about this ID, it was caused by a conref attribute; see the 
messages guide for more details.</xsl:with-param>
        <xsl:with-param name="msgnum">050</xsl:with-param>
        <xsl:with-param name="msgsev">W</xsl:with-param>
      </xsl:call-template>
    </xsl:if>
    <xsl:call-template name="checkDuplicateIds">
      <xsl:with-param name="idValues"><xsl:value-of select="normalize-space($compareTo)"/></xsl:with-param> 
    </xsl:call-template>
  </xsl:if>
</xsl:template>

<!-- Copy every node to the output. If this is a topic, check for duplicate IDs within the topic. -->
<xsl:template match="*|@*|text()|comment()|processing-instruction()" mode="verify-and-output">
  <xsl:if test="contains(@class,' topic/topic ')">
    <xsl:variable name="idStrings">
      <!-- Select every non-topic child of this element, along with descendants of
           those non-topic elements; output @id for any element with an ID. -->
      <xsl:for-each select="*[not(contains(@class,' topic/topic '))]/descendant-or-self::*[@id]">
        <xsl:text> </xsl:text><xsl:value-of select="@id"/><xsl:text> </xsl:text>
      </xsl:for-each>
    </xsl:variable>
    <xsl:call-template name="checkDuplicateIds">
      <xsl:with-param name="idValues"><xsl:value-of select="normalize-space($idStrings)"/></xsl:with-param> 
    </xsl:call-template>
  </xsl:if>
  <xsl:copy>
    <xsl:apply-templates select="*|@*|comment()|processing-instruction()|text()" mode="verify-and-output"/>
  </xsl:copy>
</xsl:template>

<!-- If the target element does not exist, this template will be called to issue an error -->
<xsl:template name="missing-target-error">
  <xsl:call-template name="output-message">
    <xsl:with-param name="msg">Unable to find target for conref="<xsl:value-of select="@conref"/>". Check to make
sure that the target element is available, and that it is a <xsl:value-of select="name()"/> element.</xsl:with-param>
    <xsl:with-param name="msgnum">030</xsl:with-param>
    <xsl:with-param name="msgsev">E</xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!-- If an ID is duplicated, and there are 2 possible targets, issue a warning -->
<xsl:template name="duplicateConrefTarget">
  <xsl:call-template name="output-message">
    <xsl:with-param name="msg">There is more than one possible target for conref="<xsl:value-of select="@conref"/>".
Only the first will be used. Remove the duplicate ID from one of the targets.</xsl:with-param>
    <xsl:with-param name="msgnum">049</xsl:with-param>
    <xsl:with-param name="msgsev">W</xsl:with-param>
  </xsl:call-template>
</xsl:template>

<!-- Determine the relative path to a conref'ed file. Start with the path and
     filename. Output each single directory, and chop it off. Keep going until
     only the filename is left. -->
<xsl:template name="find-relative-path">
  <xsl:param name="remainingpath"><xsl:value-of select="substring-before(@conref,'#')"/></xsl:param>
  <xsl:if test="contains($remainingpath,'/')">
    <xsl:value-of select="substring-before($remainingpath,'/')"/>/<xsl:text/>
    <xsl:call-template name="find-relative-path">
      <xsl:with-param name="remainingpath"><xsl:value-of select="substring-after($remainingpath,'/')"/></xsl:with-param>
    </xsl:call-template>
  </xsl:if>
</xsl:template>

<!-- 20040727: targetTopic contains the domains value from the topic that this conref points to. 
               The domains value of the target must be equal to, or a subset of, the domains value 
               of the referencing topic. If it is equal, they allow the same elements. If the target
               is a subset, it allows fewer elements, and is valid in the source. If the target is a
               superset, it may contain elements that are not valid in the source, so do not allow. -->
<xsl:template name="compareDomains">
  <xsl:param name="sourceTopic"/>
  <xsl:param name="targetTopic"/>
  <xsl:choose>
    <xsl:when test="$sourceTopic=$targetTopic or contains($sourceTopic,$targetTopic)">match</xsl:when>
    <xsl:when test="string-length($targetTopic)=0">match</xsl:when>
    <xsl:otherwise>
      <xsl:variable name="firstDomain"><xsl:value-of select="normalize-space(substring-before($targetTopic,')'))"/>)</xsl:variable>
      <xsl:variable name="otherDomains"><xsl:value-of select="substring-after($targetTopic,')')"/></xsl:variable>
      <xsl:choose>
        <xsl:when test="contains($sourceTopic,$firstDomain)">
          <xsl:call-template name="compareDomains">
            <xsl:with-param name="sourceTopic" select="$sourceTopic"/>
            <xsl:with-param name="targetTopic" select="normalize-space($otherDomains)"/>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="output-message">
            <xsl:with-param name="msg">When you conref another topic or an item in another topic, the
domains attribute of the target topic must be equal to or a subset of the current topic's
domains attribute. See the messages guide for more details.</xsl:with-param>
            <xsl:with-param name="msgnum">057</xsl:with-param>
            <xsl:with-param name="msgsev">W</xsl:with-param>
          </xsl:call-template>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!--if something has a conref attribute, jump to the target if valid and continue applying templates-->
<xsl:template match="*[@conref][@conref!='']" priority="10">
  <!-- If we have already followed a relative path, pick it up -->
  <xsl:param name="current-relative-path"/>
  <xsl:variable name="add-relative-path">
    <xsl:call-template name="find-relative-path"/>
  </xsl:variable>
  <!-- Keep the source node in a variable, to pass to the target. It can be used to save 
       attributes that were specified locally. If for some reason somebody passes from
       conref straight to conref, then just save the first one (in source-element) -->
  <xsl:param name="source-element"/>
  <xsl:variable name="source-node">
    <xsl:choose>
      <xsl:when test="$source-element"><xsl:copy-of select="$source-element"/></xsl:when>
      <xsl:otherwise><xsl:copy-of select="."/></xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <!--get element local name, parent topic's domains, and then file name, topic id, element id from conref value-->
  <xsl:variable name="element">
    <xsl:value-of select="local-name(.)"/>
  </xsl:variable>
  <xsl:variable name="domains">
      <xsl:value-of select="ancestor-or-self::*[@domains][1]/@domains"/>
  </xsl:variable>
  <xsl:variable name="file">
  <xsl:choose>
     <xsl:when test="contains(@conref,'#')"><xsl:value-of select="$WORKDIR"/><xsl:value-of select="$current-relative-path"/><xsl:value-of select="substring-before(@conref,'#')"/></xsl:when>
     <xsl:otherwise><xsl:value-of select="$WORKDIR"/><xsl:value-of select="$current-relative-path"/><xsl:value-of select="@conref"/></xsl:otherwise>
  </xsl:choose>
  </xsl:variable>

  <xsl:variable name="topicid">
  <xsl:choose>
     <xsl:when test="contains(@conref,'#') and contains(substring-after(@conref,'#'),'/')"><xsl:value-of select="substring-before(substring-after(@conref,'#'),'/')"/></xsl:when>
     <xsl:when test="contains(@conref,'#')"><xsl:value-of select="substring-after(@conref,'#')"/></xsl:when>
     <xsl:otherwise>#none#</xsl:otherwise>
  </xsl:choose>
  </xsl:variable>

  <xsl:variable name="elemid">
    <xsl:choose>
     <xsl:when test="contains(@conref,'#') and contains(substring-after(@conref,'#'),'/')"><xsl:value-of select="substring-after(substring-after(@conref,'#'),'/')"/></xsl:when>
     <xsl:otherwise>#none#</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>

  <xsl:variable name="topicpos">
  <xsl:choose>
     <xsl:when test="starts-with(@conref,'#')">samefile</xsl:when>
     <xsl:when test="contains(@conref,'#')">otherfile</xsl:when>
     <xsl:otherwise>firstinfile</xsl:otherwise>
  </xsl:choose>
  </xsl:variable>

 <!-- 20040331 RDA: Comment this out, it usually just fills up the log -->
 <!--<xsl:if test="$DBG='yes'">
  <xsl:call-template name="output-message">
   <xsl:with-param name="msg">Conref trace: </xsl:with-param>
   <xsl:with-param name="msgnum">000</xsl:with-param>
   <xsl:with-param name="msgsev">I</xsl:with-param>
  </xsl:call-template>
  <xsl:message> Element: <xsl:value-of select="name()"/>:<xsl:value-of select="$element"/></xsl:message>
  <xsl:message> Domains: <xsl:value-of select="$domains"/></xsl:message>
  <xsl:message> Working dir: <xsl:value-of select="$WORKDIR"/></xsl:message>
  <xsl:message> Project dir: <xsl:value-of select="$PROJDIR"/></xsl:message>
  <xsl:message> File: <xsl:value-of select="$file"/></xsl:message>
  <xsl:message> Topicid: <xsl:value-of select="$topicid"/></xsl:message>
  <xsl:message> Elemid: <xsl:value-of select="$elemid"/></xsl:message>
  <xsl:message> Topic location: <xsl:value-of select="$topicpos"/></xsl:message>
  <xsl:message> Source Element is <xsl:value-of select="name()"/></xsl:message>
  <xsl:message> Source File is <xsl:value-of select="@xtrf"/></xsl:message>
  <xsl:message> Source Count is <xsl:number level="any"/></xsl:message>
  <xsl:message> Already followed path to get here: <xsl:value-of select="$current-relative-path"/></xsl:message>
  <xsl:message> Relative Path to target is <xsl:value-of select="$add-relative-path"/></xsl:message>
 </xsl:if>-->

  <!-- E003262: Add (element)[1] to each apply-templates rule. This ensures that if there are 2 elements with the
                same ID in the same topic, only one will get processed. -->
  <xsl:choose>
    <!--targetting an element inside a topic-->
    <xsl:when test="contains(substring-after(@conref,'#'),'/')">
      <xsl:choose>
        <xsl:when test="$topicpos='samefile'">
          <xsl:choose>
            <xsl:when test="//*[local-name()=$element][@id=$elemid][ancestor::*[contains(@class, ' topic/topic ')][1][@id=$topicid]]">
              <xsl:variable name="testDomains">
                <xsl:call-template name="compareDomains">
                    <xsl:with-param name="sourceTopic" select="$domains"/>
                    <xsl:with-param name="targetTopic" select="//*[contains(@class, ' topic/topic ')][@id=$topicid][1]/@domains"/>
                </xsl:call-template>
              </xsl:variable>
              <xsl:if test="$testDomains='match'">
                <xsl:apply-templates select="(//*[local-name()=$element][@id=$elemid][ancestor::*[contains(@class, ' topic/topic ')][1][@id=$topicid]])[1]" mode="conref-target">
                  <xsl:with-param name="source-element"><xsl:copy-of select="$source-node"/></xsl:with-param>
                  <xsl:with-param name="current-relative-path"><xsl:value-of select="$current-relative-path"/><xsl:value-of select="$add-relative-path"/></xsl:with-param>
                </xsl:apply-templates>
                <xsl:if test="(//*[local-name()=$element][@id=$elemid][ancestor::*[contains(@class, ' topic/topic ')][1][@id=$topicid]])[2]">
                  <xsl:call-template name="duplicateConrefTarget"/>
                </xsl:if>
              </xsl:if>
            </xsl:when>
            <xsl:otherwise><xsl:call-template name="missing-target-error"/></xsl:otherwise>
          </xsl:choose>
        </xsl:when>
        <xsl:when test="$topicpos='otherfile'">
          <xsl:choose>
            <xsl:when test="document($file)//*[local-name()=$element][@id=$elemid][ancestor::*[contains(@class, ' topic/topic ')][1][@id=$topicid]]">
              <xsl:variable name="testDomains">
                <xsl:call-template name="compareDomains">
                  <xsl:with-param name="sourceTopic" select="$domains"/>
                  <xsl:with-param name="targetTopic" select="document($file)//*[contains(@class, ' topic/topic ')][@id=$topicid][1]/@domains"/>
                </xsl:call-template>
              </xsl:variable>
              <xsl:if test="$testDomains='match'">
                <xsl:apply-templates select="(document($file)//*[local-name()=$element][@id=$elemid][ancestor::*[contains(@class, ' topic/topic ')][1][@id=$topicid]])[1]" mode="conref-target">
                  <xsl:with-param name="source-element"><xsl:copy-of select="$source-node"/></xsl:with-param>
                  <xsl:with-param name="current-relative-path"><xsl:value-of select="$current-relative-path"/><xsl:value-of select="$add-relative-path"/></xsl:with-param>
                </xsl:apply-templates>
                <xsl:if test="(document($file)//*[local-name()=$element][@id=$elemid][ancestor::*[contains(@class, ' topic/topic ')][1][@id=$topicid]])[2]">
                  <xsl:call-template name="duplicateConrefTarget"/>
                </xsl:if>
              </xsl:if>
            </xsl:when>
            <xsl:otherwise><xsl:call-template name="missing-target-error"/></xsl:otherwise>
          </xsl:choose>
        </xsl:when>
        <xsl:otherwise/><!--never happens - only other value is firstinfile, but we know there's a # in the conref so it's either samefile or otherfile-->
      </xsl:choose>
    </xsl:when>

    <!--targetting a topic-->
    <xsl:when test="contains(@class, ' topic/topic ')">
      <xsl:choose>
        <xsl:when test="$topicpos='samefile'">
          <xsl:choose>
            <xsl:when test="//*[contains(@class, ' topic/topic ')][@id=$topicid][local-name()=$element]">
              <xsl:variable name="testDomains">
                <xsl:call-template name="compareDomains">
                  <xsl:with-param name="sourceTopic" select="$domains"/>
                  <xsl:with-param name="targetTopic" select="//*[contains(@class, ' topic/topic ')][@id=$topicid][1][local-name()=$element]/@domains"/>
                </xsl:call-template>
              </xsl:variable>
              <xsl:if test="$testDomains='match'">
                <xsl:apply-templates select="(//*[contains(@class, ' topic/topic ')][@id=$topicid][local-name()=$element])[1]" mode="conref-target">
                  <xsl:with-param name="source-element"><xsl:copy-of select="$source-node"/></xsl:with-param>
                  <xsl:with-param name="current-relative-path"><xsl:value-of select="$current-relative-path"/><xsl:value-of select="$add-relative-path"/></xsl:with-param>
                </xsl:apply-templates>
                <xsl:if test="(//*[contains(@class, ' topic/topic ')][@id=$topicid][local-name()=$element])[2]">
                  <xsl:call-template name="duplicateConrefTarget"/>
                </xsl:if>
              </xsl:if>
            </xsl:when>
            <xsl:otherwise><xsl:call-template name="missing-target-error"/></xsl:otherwise>
          </xsl:choose>
        </xsl:when>
        <xsl:when test="$topicpos='otherfile'">
          <xsl:choose>
            <xsl:when test="document($file)//*[contains(@class, ' topic/topic ')][@id=$topicid][local-name()=$element]">
              <xsl:variable name="testDomains">
                <xsl:call-template name="compareDomains">
                  <xsl:with-param name="sourceTopic" select="$domains"/>
                  <xsl:with-param name="targetTopic" select="document($file)//*[contains(@class, ' topic/topic ')][@id=$topicid][1][local-name()=$element]/@domains"/>
                </xsl:call-template>
              </xsl:variable>
              <xsl:if test="$testDomains='match'">
                <xsl:apply-templates select="(document($file)//*[contains(@class, ' topic/topic ')][@id=$topicid][local-name()=$element])[1]" mode="conref-target">
                  <xsl:with-param name="source-element"><xsl:copy-of select="$source-node"/></xsl:with-param>
                  <xsl:with-param name="current-relative-path"><xsl:value-of select="$current-relative-path"/><xsl:value-of select="$add-relative-path"/></xsl:with-param>
                </xsl:apply-templates>
                <xsl:if test="(document($file)//*[contains(@class, ' topic/topic ')][@id=$topicid][local-name()=$element])[2]">
                  <xsl:call-template name="duplicateConrefTarget"/>
                </xsl:if>
              </xsl:if>
            </xsl:when>
            <xsl:otherwise><xsl:call-template name="missing-target-error"/></xsl:otherwise>
          </xsl:choose>
        </xsl:when>
        <xsl:when test="$topicpos='firstinfile'">
          <xsl:choose>
            <xsl:when test="document($file)//*[contains(@class, ' topic/topic ')][1][local-name()=$element]">
              <xsl:variable name="testDomains">
                <xsl:call-template name="compareDomains">
                  <xsl:with-param name="sourceTopic" select="$domains"/>
                  <xsl:with-param name="targetTopic" select="document($file)//*[contains(@class, ' topic/topic ')][1][local-name()=$element]/@domains"/>
                </xsl:call-template>
              </xsl:variable>
              <xsl:if test="$testDomains='match'">
                <xsl:apply-templates select="(document($file)//*[contains(@class, ' topic/topic ')][1][local-name()=$element])[1]" mode="conref-target">
                  <xsl:with-param name="source-element"><xsl:copy-of select="$source-node"/></xsl:with-param>
                  <xsl:with-param name="current-relative-path"><xsl:value-of select="$current-relative-path"/><xsl:value-of select="$add-relative-path"/></xsl:with-param>
                </xsl:apply-templates>
                <xsl:if test="(document($file)//*[contains(@class, ' topic/topic ')][1][local-name()=$element])[2]">
                  <xsl:call-template name="duplicateConrefTarget"/>
                </xsl:if>
              </xsl:if>
            </xsl:when>
            <xsl:otherwise><xsl:call-template name="missing-target-error"/></xsl:otherwise>
          </xsl:choose>
        </xsl:when>
        <xsl:otherwise/><!--never happens - only three possible values for topicpos, all are tested-->
      </xsl:choose>
    </xsl:when>

    <!--targetting a topicref from within a map-->
    <xsl:when test="contains(@class, ' map/topicref ')">
      <xsl:choose>
        <xsl:when test="$topicpos='samefile'">
          <xsl:choose>
            <xsl:when test="//*[contains(@class, ' map/topicref ')][@id=$topicid][local-name()=$element]">
              <xsl:apply-templates select="(//*[contains(@class, ' map/topicref ')][@id=$topicid][local-name()=$element])[1]" mode="conref-target">
                <xsl:with-param name="source-element"><xsl:copy-of select="$source-node"/></xsl:with-param>
                <xsl:with-param name="current-relative-path"><xsl:value-of select="$current-relative-path"/><xsl:value-of select="$add-relative-path"/></xsl:with-param>
              </xsl:apply-templates>
              <xsl:if test="(//*[contains(@class, ' map/topicref ')][@id=$topicid][local-name()=$element])[2]">
                <xsl:call-template name="duplicateConrefTarget"/>
              </xsl:if>
            </xsl:when>
            <xsl:otherwise><xsl:call-template name="missing-target-error"/></xsl:otherwise>
          </xsl:choose>
        </xsl:when>
        <xsl:when test="$topicpos='otherfile'">
          <xsl:choose>
            <xsl:when test="document($file)//*[contains(@class, ' map/topicref ')][@id=$topicid][local-name()=$element]">
              <xsl:apply-templates select="(document($file)//*[contains(@class, ' map/topicref ')][@id=$topicid][local-name()=$element])[1]" mode="conref-target">
                <xsl:with-param name="source-element"><xsl:copy-of select="$source-node"/></xsl:with-param>
                <xsl:with-param name="current-relative-path"><xsl:value-of select="$current-relative-path"/><xsl:value-of select="$add-relative-path"/></xsl:with-param>
              </xsl:apply-templates>
              <xsl:if test="(document($file)//*[contains(@class, ' map/topicref ')][@id=$topicid][local-name()=$element])[2]">
                <xsl:call-template name="duplicateConrefTarget"/>
              </xsl:if>
            </xsl:when>
            <xsl:otherwise><xsl:call-template name="missing-target-error"/></xsl:otherwise>
          </xsl:choose>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="output-message">
            <xsl:with-param name="msg">You must provide the id of the target topicref you want to reuse.
For example, mymap.ditamap#mytopicrefid.</xsl:with-param>
            <xsl:with-param name="msgnum">012</xsl:with-param>
            <xsl:with-param name="msgsev">E</xsl:with-param>
          </xsl:call-template>
	</xsl:otherwise>
      </xsl:choose>
    </xsl:when>

    <!--targetting anything else within a map (such as reltable)-->
    <xsl:when test="contains(/*/@class, ' map/map ')">
      <xsl:choose>
        <xsl:when test="$topicpos='samefile'">
          <xsl:choose>
            <xsl:when test="//*[@id=$topicid][local-name()=$element]">
              <xsl:apply-templates select="(//*[@id=$topicid][local-name()=$element])[1]" mode="conref-target">
                <xsl:with-param name="source-element"><xsl:copy-of select="$source-node"/></xsl:with-param>
                <xsl:with-param name="current-relative-path"><xsl:value-of select="$current-relative-path"/><xsl:value-of select="$add-relative-path"/></xsl:with-param>
              </xsl:apply-templates>
              <xsl:if test="(//*[@id=$topicid][local-name()=$element])[2]">
                <xsl:call-template name="duplicateConrefTarget"/>
              </xsl:if>
            </xsl:when>
            <xsl:otherwise><xsl:call-template name="missing-target-error"/></xsl:otherwise>
          </xsl:choose>
        </xsl:when>
        <xsl:when test="$topicpos='otherfile'">
          <xsl:choose>
            <xsl:when test="document($file)//*[@id=$topicid][local-name()=$element]">
              <xsl:apply-templates select="(document($file)//*[@id=$topicid][local-name()=$element])[1]" mode="conref-target">
                <xsl:with-param name="source-element"><xsl:copy-of select="$source-node"/></xsl:with-param>
                <xsl:with-param name="current-relative-path"><xsl:value-of select="$current-relative-path"/><xsl:value-of select="$add-relative-path"/></xsl:with-param>
              </xsl:apply-templates>
              <xsl:if test="(document($file)//*[@id=$topicid][local-name()=$element])[2]">
                <xsl:call-template name="duplicateConrefTarget"/>
              </xsl:if>
            </xsl:when>
            <xsl:otherwise><xsl:call-template name="missing-target-error"/></xsl:otherwise>
          </xsl:choose>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="output-message">
            <xsl:with-param name="msg">You must provide the id of the target topicref you want to reuse.
For example, mymap.ditamap#mytopicrefid.</xsl:with-param>
            <xsl:with-param name="msgnum">012</xsl:with-param>
            <xsl:with-param name="msgsev">E</xsl:with-param>
          </xsl:call-template>
	</xsl:otherwise>
      </xsl:choose>
    </xsl:when>
    
    <xsl:otherwise>
      <xsl:call-template name="output-message">
        <xsl:with-param name="msg">Incorrectly formed conref attribute: <xsl:value-of select="@conref"/>
Make sure the syntax is correct and try again. </xsl:with-param>
    	<xsl:with-param name="msgnum">014</xsl:with-param>
        <xsl:with-param name="msgsev">E</xsl:with-param>
      </xsl:call-template>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- When an element is the target of a conref, treat everything the same as any other element EXCEPT the attributes.
     Create the current element, and add all attributes except @id. Then go back to the original element
     (passed in source-element). Process all attributes on that element, except @conref. They will
     replace values in the source. -->
<xsl:template match="*" mode="conref-target">
  <xsl:param name="source-element"/>
  <xsl:param name="current-relative-path"/> <!-- File system path from original file to here -->
  
  <xsl:choose>
    <!-- If for some bizarre reason you conref to another element that uses @conref, forget the original and continue here. -->
    <xsl:when test="@conref">
      <xsl:apply-templates select=".">
        <xsl:with-param name="source-element"><xsl:copy-of select="$source-element"/></xsl:with-param>
        <xsl:with-param name="current-relative-path"><xsl:value-of select="$current-relative-path"/></xsl:with-param>
      </xsl:apply-templates>
    </xsl:when>
    <xsl:otherwise>
      <xsl:copy>                                                <!-- Create the start tag, and add @* except @id -->
        <xsl:apply-templates select="@*[not(name()='id')]">
          <xsl:with-param name="current-relative-path"><xsl:value-of select="$current-relative-path"/></xsl:with-param>
        </xsl:apply-templates>
    
        <!-- If any information was passed in to get us back to the original element, go back and save @* -->
        <xsl:apply-templates select="$source-element" mode="original-attributes"/>
        
        <!-- Continue processing this element as any other -->
        <xsl:apply-templates select="*|comment()|processing-instruction()|text()">
          <xsl:with-param name="current-relative-path"><xsl:value-of select="$current-relative-path"/></xsl:with-param>
        </xsl:apply-templates>
      </xsl:copy>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Processing a copy of the original element, that used @conref: apply-templates on
     all of the attributes, though some may be filtered out. -->
<xsl:template match="*" mode="original-attributes">
  <xsl:apply-templates select="@*" mode="original-attributes"/>
</xsl:template>
<xsl:template match="@*" mode="original-attributes">
  <xsl:copy><xsl:value-of select="."/></xsl:copy>
</xsl:template>

<!-- If an element is required, it must be specified on the original source element to avoid parsing errors.
     Such attributes should NOT be copied from the source. Conref should also not be copied. 
     NOTE: if a new specialized element requires attributes, it should be added here. -->
<!-- E003898: debug attrs xtrf, xtrc should continue to point to the referencing element, not the target. -->
<xsl:template match="@xtrc|@xtrf" mode="original-attributes" priority="10"/>
<xsl:template match="@conref" mode="original-attributes" priority="10"/>
<xsl:template match="*[contains(@class,' topic/image ')]/@href" mode="original-attributes" priority="10"/>
<xsl:template match="*[contains(@class,' topic/tgroup ')]/@cols" mode="original-attributes" priority="10"/>
<xsl:template match="*[contains(@class,' topic/boolean ')]/@state" mode="original-attributes" priority="10"/>
<xsl:template match="*[contains(@class,' topic/state ')]/@name" mode="original-attributes" priority="10"/>
<xsl:template match="*[contains(@class,' topic/state ')]/@value" mode="original-attributes" priority="10"/>
<!-- topichead is specialized from topicref, and requires @navtitle -->
<xsl:template match="*[contains(@class,' map/topichead ')]/@navtitle" mode="original-attributes" priority="10"/>

<!-- 20031231 Adjust relative paths when an HREF is in a file, as part of a conref'ed section -->
<!-- 20040229 If the HREF is to a spot in the current file, it must now include the file name;
              If it is external, we should not add names or paths. -->
<xsl:template match="@href">
  <xsl:param name="current-relative-path"/>
  <xsl:attribute name="href">
    <xsl:choose>
      <xsl:when test="../@scope='external'"><xsl:value-of select="."/></xsl:when>
      <xsl:when test="starts-with(.,'#')"><xsl:value-of select="."/></xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$current-relative-path"/><xsl:value-of select="."/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:attribute>
</xsl:template>

<!--copy everything else-->
<xsl:template match="*|@*|comment()|processing-instruction()|text()">
  <xsl:param name="current-relative-path"/>
  <xsl:copy>
    <xsl:apply-templates select="*|@*|comment()|processing-instruction()|text()">
      <xsl:with-param name="current-relative-path"><xsl:value-of select="$current-relative-path"/></xsl:with-param>
    </xsl:apply-templates>
  </xsl:copy>
</xsl:template>

</xsl:stylesheet>
