<?xml version="1.0"?>

<!-- 
     Conversion from DITA map or maplist to .NET Help contents
     Input = one DITA map file, or a maplist pointing to multiple maps
     Output = one HxT contents file for use with the .NET help
     
     Options:
        /OUTEXT  = XHTML output extension (default is 'html')
        /WORKDIR = The working directory that contains the document being transformed.
                   Needed as a directory prefix for the @href "document()" function calls. 
                   Default is './'

     Updates:
        Rel 3.8: Add support for copy-to attribute in maps
        3.8.2:   Allow for paths in maplist
-->

<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                version="1.0">

<!-- Include error message template -->
<xsl:include href="output-message.xsl"/>

<xsl:output method="xml"
            encoding="utf-8"
            doctype-system="ms-help://hx/resources/HelpTOC.DTD"
/>

<!-- Set the prefix for error message numbers -->
<xsl:variable name="msgprefix">IDXS</xsl:variable>
<xsl:variable name="newline"><xsl:text>
</xsl:text></xsl:variable>

<!-- *************************** Command line parameters *********************** -->
<xsl:param name="OUTEXT" select="'html'"/><!-- "htm" and "html" are valid values -->
<xsl:param name="WORKDIR" select="'./'"/>

<!-- *********************************************************************************
     Setup the HelpTOC wrapper for the table of contents
     ********************************************************************************* -->
<xsl:template match="/">
  <HelpTOC DTDVersion="1.0">
    <xsl:value-of select="$newline"/>
    <xsl:apply-templates/>
    <xsl:value-of select="$newline"/>
  </HelpTOC>
</xsl:template>

<!-- *********************************************************************************
     Process each of the maps in the navigation section.
     ********************************************************************************* -->
<xsl:template match="/maplist">
  <xsl:apply-templates select="nav/map"/>
</xsl:template>

<!-- *********************************************************************************
     ********************************************************************************* -->
<xsl:template match="/*[contains(@class, ' map/map ')]">
  <xsl:param name="pathFromMaplist"/>
  <xsl:apply-templates select="*[contains(@class, ' map/topicref ')][not(@toc='no')]">
    <xsl:with-param name="pathFromMaplist" select="$pathFromMaplist"/>
  </xsl:apply-templates>
</xsl:template>

<!-- *********************************************************************************
     For each map in the navigation secton of the maplist, process the contents.
     ********************************************************************************* -->
<xsl:template match="nav/map">
  <xsl:variable name="MapWithPath"><xsl:value-of select="$WORKDIR"/><xsl:value-of select="@file"/></xsl:variable>
  <xsl:apply-templates select="document($MapWithPath)/*[contains(@class, ' map/map ')]">
    <xsl:with-param name="pathFromMaplist"><xsl:call-template name="getRelativePath"/></xsl:with-param>
  </xsl:apply-templates>
</xsl:template>

<!-- *********************************************************************************
     ********************************************************************************* -->
<xsl:template match="*[contains(@class, ' map/topicref ')]">
  <xsl:param name="pathFromMaplist"/>
  <xsl:choose>
    <xsl:when test="@toc='no'"/>
    <xsl:when test="not(@href)">
      <!-- Can you just have container elements in .net? For now, process if there 
           is a title, or if there are children. -->
      <xsl:if test="*|@navtitle">
        <xsl:call-template name="output-toc-entry">
          <xsl:with-param name="pathFromMaplist" select="$pathFromMaplist"/>
        </xsl:call-template>
      </xsl:if>
    </xsl:when>
    <xsl:when test="not(@format)">
      <xsl:call-template name="output-toc-entry">
        <xsl:with-param name="pathFromMaplist" select="$pathFromMaplist"/>
      </xsl:call-template>
    </xsl:when>
    <xsl:when test="@format='dita' or @format='DITA'">
      <xsl:call-template name="output-toc-entry">
        <xsl:with-param name="pathFromMaplist" select="$pathFromMaplist"/>
      </xsl:call-template>
    </xsl:when>
    <xsl:when test="(contains(@format,'htm') or contains(@format,'HTM')) and (@scope='external' or @scope='peer')">
      <xsl:call-template name="output-toc-entry">
        <xsl:with-param name="pathFromMaplist" select="$pathFromMaplist"/>
      </xsl:call-template>
    </xsl:when>
    <!-- If we compile .net in the working dir, these will not be available -->
    <xsl:when test="contains(@format,'htm') or contains(@format,'HTM')">
      <xsl:call-template name="output-toc-entry">
        <xsl:with-param name="pathFromMaplist" select="$pathFromMaplist"/>
      </xsl:call-template>
    </xsl:when>
    <xsl:otherwise>
      <!-- @format is not DITA and not HTML: can it be included in .net? -->
      <xsl:call-template name="output-toc-entry">
        <xsl:with-param name="pathFromMaplist" select="$pathFromMaplist"/>
      </xsl:call-template>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>
    
<xsl:template name="output-toc-entry">
  <xsl:param name="pathFromMaplist"/>
  <xsl:variable name="title-attribute"><xsl:call-template name="get-title-attribute"/></xsl:variable>
  <xsl:choose>
    <xsl:when test="not(@href) and string-length($title-attribute)=0">
      <xsl:apply-templates select="*[contains(@class, ' map/topicref ')][not(contains(@toc,'no'))]"/>
    </xsl:when>
    <xsl:otherwise>
      <xsl:value-of select="$newline"/>
      <HelpTOCNode>
        <xsl:choose>
          <xsl:when test="string-length($title-attribute)>0">
            <xsl:attribute name="Title"><xsl:value-of select="$title-attribute"/></xsl:attribute>
          </xsl:when>
          <xsl:otherwise>
            <xsl:attribute name="Title"><xsl:text> </xsl:text></xsl:attribute>
          </xsl:otherwise>
        </xsl:choose>
    
        <xsl:if test="@href">
          <xsl:attribute name="Url">
            <xsl:choose>        <!-- What if targeting a nested topic? Need to keep the ID? -->
              <xsl:when test="contains(@copy-to,'.dita')">
                <xsl:if test="not(@scope='external')"><xsl:value-of select="$pathFromMaplist"/></xsl:if>
                <xsl:value-of select="substring-before(@copy-to,'.dita')"/>.<xsl:value-of select="$OUTEXT"/>
              </xsl:when>
              <xsl:when test="contains(@href,'.dita')">
                <xsl:if test="not(@scope='external')"><xsl:value-of select="$pathFromMaplist"/></xsl:if>
                <xsl:value-of select="substring-before(@href,'.dita')"/>.<xsl:value-of select="$OUTEXT"/>
              </xsl:when>
              <xsl:otherwise>  <!-- If non-DITA, keep the href as-is -->
                <xsl:if test="not(@scope='external')"><xsl:value-of select="$pathFromMaplist"/></xsl:if>
                <xsl:value-of select="@href"/>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:attribute>
        </xsl:if>
     
        <!-- Process any children that should be in the TOC -->
        <xsl:apply-templates select="*[contains(@class, ' map/topicref ')][not(contains(@toc,'no'))]">
          <xsl:with-param name="pathFromMaplist" select="$pathFromMaplist"/>
        </xsl:apply-templates>
      </HelpTOCNode>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="get-title-attribute">
      <xsl:choose>
           
        <!-- If navtitle is specified, use it -->
        <xsl:when test="@navtitle"><xsl:value-of select="@navtitle"/></xsl:when>

        <!-- If this references a DITA file (has @href, not "local" or "external"),
             try to open the file and get the title -->
        <xsl:when test="@href and 
                        not ((ancestor-or-self::*/@scope)[last()]='external') and
                        not ((ancestor-or-self::*/@scope)[last()]='peer') and
                        not ((ancestor-or-self::*/@type)[last()]='external') and
                        not ((ancestor-or-self::*/@type)[last()]='local')">
          <!-- Need to worry about targeting a nested topic? Not for now. -->
          <xsl:variable name="FileWithPath"><xsl:value-of select="$WORKDIR"/><xsl:choose>
            <xsl:when test="@copy-to"><xsl:value-of select="@copy-to"/></xsl:when>
            <xsl:when test="contains(@href,'#')"><xsl:value-of select="substring-before(@href,'#')"/></xsl:when>
            <xsl:otherwise><xsl:value-of select="@href"/></xsl:otherwise></xsl:choose></xsl:variable>
          <xsl:variable name="TargetFile" select="document($FileWithPath)"/>
          
          <xsl:choose>
            <xsl:when test="not($TargetFile)">   <!-- DITA file does not exist -->
              <xsl:choose>
                <xsl:when test="*[contains(@class, ' map/topicmeta ')]/*[contains(@class, ' map/linktext ')]">  <!-- attempt to recover by using linktext -->
                  <xsl:value-of select="*[contains(@class, ' map/topicmeta ')]/*[contains(@class, ' map/linktext ')]"/>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:call-template name="output-message">
                    <xsl:with-param name="msg">File <xsl:value-of select="@href"/> does not exist.</xsl:with-param>
                    <xsl:with-param name="msgnum">004</xsl:with-param>
                    <xsl:with-param name="msgsev">W</xsl:with-param>
                  </xsl:call-template>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:when>
            <!-- First choice for navtitle: topic/titlealts/navtitle -->
            <xsl:when test="$TargetFile/*[contains(@class,' topic/topic ')]/*[contains(@class,' topic/titlealts ')]/*[contains(@class,' topic/navtitle ')]">
              <xsl:value-of select="$TargetFile/*[contains(@class,' topic/topic ')]/*[contains(@class,' topic/titlealts ')]/*[contains(@class,' topic/navtitle ')]"/>
            </xsl:when>
            <!-- Second choice for navtitle: topic/title -->
            <xsl:when test="$TargetFile/*[contains(@class,' topic/topic ')]/*[contains(@class,' topic/title ')]">
              <xsl:value-of select="$TargetFile/*[contains(@class,' topic/topic ')]/*[contains(@class,' topic/title ')]"/>
            </xsl:when>
            <!-- This might be a combo article; modify the same queries: dita/topic/titlealts/navtitle -->
            <xsl:when test="$TargetFile/dita/*[contains(@class,' topic/topic ')]/*[contains(@class,' topic/titlealts ')]/*[contains(@class,' topic/navtitle ')]">
              <xsl:value-of select="$TargetFile/dita/*[contains(@class,' topic/topic ')]/*[contains(@class,' topic/titlealts ')]/*[contains(@class,' topic/navtitle ')]"/>
            </xsl:when>
            <!-- Second choice: dita/topic/title -->
            <xsl:when test="$TargetFile/dita/*[contains(@class,' topic/topic ')]/*[contains(@class,' topic/title ')]">
              <xsl:value-of select="$TargetFile/dita/*[contains(@class,' topic/topic ')]/*[contains(@class,' topic/title ')]"/>
            </xsl:when>
            <!-- Last choice: use the linktext specified within the topicref -->
            <xsl:when test="*[contains(@class, ' map/topicmeta ')]/*[contains(@class, ' map/linktext ')]">
              <xsl:value-of select="*[contains(@class, ' map/topicmeta ')]/*[contains(@class, ' map/linktext ')]"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:call-template name="output-message">
                <xsl:with-param name="msg">Could not retrieve a title from <xsl:value-of select="$TargetFile"/>. Using '***'.</xsl:with-param>
                <xsl:with-param name="msgnum">005</xsl:with-param>
                <xsl:with-param name="msgsev">W</xsl:with-param>
              </xsl:call-template>
              <xsl:text>***</xsl:text>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:when>

        <!-- If there is no title and none can be retrieved, check for <linktext> -->
        <xsl:when test="*[contains(@class, ' map/topicmeta ')]/*[contains(@class, ' map/linktext ')]">
          <xsl:value-of select="*[contains(@class, ' map/topicmeta ')]/*[contains(@class, ' map/linktext ')]"/>
        </xsl:when>

        <!-- No local title, and not targeting a DITA file. Could be just a container setting
             metadata, or a file reference with no title. Issue message for the second case. -->
        <xsl:otherwise>
          <xsl:if test="@href">
              <xsl:call-template name="output-message">
                <xsl:with-param name="msg">Could not retrieve a title from <xsl:value-of select="@href"/>. Using the HREF value.</xsl:with-param>
                <xsl:with-param name="msgnum">005</xsl:with-param>
                <xsl:with-param name="msgsev">W</xsl:with-param>
              </xsl:call-template>
              <xsl:value-of select="@href"/>
          </xsl:if>
        </xsl:otherwise>
      </xsl:choose>
</xsl:template>

<!-- These are here just to prevent accidental fallthrough -->
<xsl:template match="*[contains(@class, ' map/navref ')]"/>
<xsl:template match="*[contains(@class, ' map/anchor ')]"/>
<xsl:template match="*[contains(@class, ' map/reltable ')]"/>
<xsl:template match="*[contains(@class, ' map/topicmeta ')]"/>
<xsl:template match="maplist/link"/>

<xsl:template match="*">
  <xsl:apply-templates/>
</xsl:template>

<!-- Template to get the relative path from a maplist to a map -->
<xsl:template name="getRelativePath">
  <xsl:param name="remainingPath" select="@file"/>
  <xsl:choose>
    <xsl:when test="contains($remainingPath,'/')">
      <xsl:value-of select="substring-before($remainingPath,'/')"/><xsl:text>/</xsl:text>
      <xsl:call-template name="getRelativePath">
        <xsl:with-param name="remainingPath" select="substring-after($remainingPath,'/')"/>
      </xsl:call-template>
    </xsl:when>
    <xsl:when test="contains($remainingPath,'\')">
      <xsl:value-of select="substring-before($remainingPath,'\')"/><xsl:text>/</xsl:text>
      <xsl:call-template name="getRelativePath">
        <xsl:with-param name="remainingPath" select="substring-after($remainingPath,'\')"/>
      </xsl:call-template>
    </xsl:when>
  </xsl:choose>
</xsl:template>

</xsl:stylesheet>
