<?xml version="1.0" encoding="UTF-8" ?>

<!--
    Provide an HTML summary of a DITA map.
    
    * Each topicref will become a list item in a list
       - If the parent has collection-type="sequence", the list will be ordered
       - Otherwise, the list will be unordered
    * If the topicref points to something (Rather than just containing a title), the resulting list item will be a link to that item
       - External or non-dita addresses will be left as-is
       - References to DITA files will have the extension changed to match the specified output extension
    * If available, the title of the target topic will be placed in the list item; otherwise, the HREF value is used
    * If selection attributes, @linking, @toc, or @print are specified, those will be included after the link
    * If a topicref contains other topicrefs, the children will be in a nested list
    * The anchor and navref elements will also appear as list items, containing the name of the element followed by the required attribute
    * The reltable will go into an HTML table: <table border="1" frame="border">
       - Each <relrow> will become a <tr>
       - Each <relcell> will become a <td>
       - Each <td> will contain an unordered list to hold the topicref elements; they can nest as with normal topicrefs
    * The topicmeta element will be ignored
    * If a reltable contains headers that define types, a header section will be generated for the table in the HTML that specifies those types

    P017907: Update logic to reduce memory consumption. When possible, process all children at
             once, rather than stepping through items sequentially (which builds up a large
             stack in memory). Still must step through for the root level, when using a mix
             of reltables and topicrefs, and there are topicrefs after the reltable.
-->
<xsl:stylesheet version="1.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

<xsl:include href="output-message.xsl"/>

<xsl:output method="xml"
            encoding="utf-8"
            indent="no"
            doctype-system="http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd"
            doctype-public="-//W3C//DTD XHTML 1.0 Transitional//EN"
/>

<xsl:param name="OUTEXT" select="'html'"/>
<xsl:variable name="msgprefix">IDXS</xsl:variable>
<xsl:variable name="newline"><xsl:text>
</xsl:text></xsl:variable>

<!-- This should only work on a map, not on a maplist. The map will generate the HTML structure.
     If there are no reltables, just process everything.
     If there are reltables, but they are all at the end:
        Process everything but reltables, then process the reltables.
     If the reltables are mixed in, processing is complex:
        Process the first child with mode=root.
        If the item is not a reltable, create a list, and process the item; sequentially process 
            the following elements, stopping at a reltable. Then jump to the reltable with mode=root.
        If the item is a reltable, create the table, and continue to the next item with mode=root.
        If topicmeta is found, ignore it and move on.
     -->
<xsl:template match="*[contains(@class, ' map/map ')]">
  <html>
    <head>
      <meta http-equiv="Content-Type" content="text/html; charset=utf-8" />
    </head>
    <body>
      <xsl:apply-templates select="@title"/>
      <xsl:apply-templates select="@anchorref"/>
      <!-- If there are no reltables, skip the complex processing. Otherwise, use mode=root. -->
      <xsl:choose>
        <!-- contains reltables, and something other than reltable -->
        <xsl:when test="*[contains(@class,' map/reltable ')] and *[not(contains(@class,' map/reltable ') or contains(@class,' map/topicmeta '))]">
          <xsl:choose>
            <!-- Is there any non-reltable after the first reltable? -->
            <xsl:when test="*[contains(@class,' map/reltable ')][1]/following-sibling::*[not(contains(@class,' map/reltable '))]">
              <xsl:apply-templates select="*[1]" mode="root"/>
            </xsl:when>
            <xsl:when test="@collection-type='sequence'">
                <ol><xsl:apply-templates select="*[not(contains(@class,' map/reltable '))]"/></ol>
                <xsl:apply-templates select="*[contains(@class,' map/reltable ')]"/>
            </xsl:when>
            <xsl:otherwise>
                <ul><xsl:apply-templates select="*[not(contains(@class,' map/reltable '))]"/></ul>
                <xsl:apply-templates select="*[contains(@class,' map/reltable ')]"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:when>
        <!-- contains only reltable (and maybe topicmeta) -->
        <xsl:when test="*[contains(@class,' map/reltable ')]">
            <xsl:apply-templates/>
        </xsl:when>
        <!-- No reltables, and ordered output -->
        <xsl:when test="@collection-type='sequence'">
          <ol><xsl:apply-templates/></ol>
        </xsl:when>
        <!-- No reltables, default unordered output -->
        <xsl:otherwise>
          <ul><xsl:apply-templates/></ul>
        </xsl:otherwise>
      </xsl:choose>
    </body>
  </html>
</xsl:template>

<!-- Topicmeta is ignored for this. If found with mode="root", move to the next element. -->
<xsl:template match="*[contains(@class, ' map/topicmeta ')]"/>
<xsl:template match="*[contains(@class, ' map/topicmeta ')]" mode="root">
  <xsl:apply-templates select="following-sibling::*[1]" mode="root"/>
</xsl:template>

<!-- If the map has a title, it will become the heading for the preview. -->
<xsl:template match="*[contains(@class, ' map/map ')]/@title">
	<h1><xsl:value-of select="."/></h1>
</xsl:template>

<!-- If there is an anchorref attribute, include it so that users can see it. -->
<xsl:template match="*[contains(@class, ' map/map ')]/@anchorref">
  <p>Anchorref: <xsl:value-of select="."/></p>
</xsl:template>

<!-- Turn the reltable into a table, and process children. Then move to the next
     item with mode="root", which means another list will be created for any
     following topics, or a new table for the next reltable. -->
<xsl:template match="*[contains(@class, ' map/reltable ')]" mode="root">
  <xsl:apply-templates select="."/>
  <xsl:apply-templates select="following-sibling::*[1]" mode="root"/>
</xsl:template>
<xsl:template match="*[contains(@class, ' map/reltable ')]">
  <table border="1" frame="border">
    <xsl:apply-templates/>
  </table>
</xsl:template>

<xsl:template match="*[contains(@class, ' map/relheader ')]">
  <xsl:value-of select="$newline"/>
  <tr>
    <xsl:apply-templates/>
  </tr>
</xsl:template>

<!-- If there are colspecs, turn them into headers, containing any attributes
     that we put out for normal topicrefs. -->
<xsl:template match="*[contains(@class, ' map/relcolspec ')]">
  <th><xsl:apply-templates select="@*"/></th>
</xsl:template>

<xsl:template match="*[contains(@class, ' map/relrow ')]">
  <xsl:value-of select="$newline"/>
  <tr valign="top">
    <xsl:apply-templates/>
  </tr>
</xsl:template>

<!-- Place the children in a list based on @collection-type -->
<xsl:template match="*[contains(@class, ' map/relcell ')]">
  <xsl:value-of select="$newline"/>
  <td>
      <xsl:choose>
        <xsl:when test="*[not(contains(@class,' map/topicmeta '))] and @collection-type='sequence'">
          <ol><xsl:apply-templates/></ol>
        </xsl:when>
        <xsl:when test="*[not(contains(@class,' map/topicmeta '))]">
          <ul><xsl:apply-templates/></ul>
        </xsl:when>
        <xsl:otherwise>
          <xsl:comment> space </xsl:comment>
        </xsl:otherwise>
      </xsl:choose>
  </td>
</xsl:template>

<!-- The root mode indicates that there is not a list open for the current level. So,
     open the list, and then process the current element, followed by everything up
     to the next reltable. Then process the next reltable, if one exists. -->
<xsl:template match="*" mode="root">
  <xsl:choose>
    <xsl:when test="../@collection-type='sequence'">
      <ol>
          <xsl:apply-templates select="."/>
          <xsl:apply-templates select="following-sibling::*[1]" mode="up-to-reltable"/>
      </ol>
    </xsl:when>
    <xsl:otherwise>
      <ul>
          <xsl:apply-templates select="."/>
          <xsl:apply-templates select="following-sibling::*[1]" mode="up-to-reltable"/>
      </ul>
    </xsl:otherwise>
  </xsl:choose>
  <xsl:apply-templates select="following-sibling::*[contains(@class, ' map/reltable ')][1]" mode="root"/>
</xsl:template>

<!-- Match anything but a reltable at the root level; process it, and continue sequentially
     until a reltable is found -->
<xsl:template match="*" mode="up-to-reltable">
    <xsl:apply-templates select="."/>
    <xsl:apply-templates select="following-sibling::*[1]" mode="up-to-reltable"/>
</xsl:template>
<xsl:template match="*[contains(@class, ' map/reltable ')]" mode="up-to-reltable"/>

<!-- Output the navref, with the required mapref attribute, as a list item -->
<xsl:template match="*[contains(@class, ' map/navref ')]">
  <xsl:value-of select="$newline"/>
  <li>Navref: mapref="<xsl:value-of select="@mapref"/>"</li>
</xsl:template>

<!-- Output the anchor, with the required id attribute, as a list item -->
<xsl:template match="*[contains(@class, ' map/anchor ')]">
  <xsl:value-of select="$newline"/>
  <li>Anchor: id="<xsl:value-of select="@id"/>"</li>
</xsl:template>

<!-- Get the topicref title. It's done in two different cases, to make it a named template. -->
<xsl:template name="get-topicref-title">
  <xsl:choose>
    <!-- First choice: use the navtitle attribute -->
    <xsl:when test="@navtitle"><xsl:value-of select="@navtitle"/></xsl:when>
    <!-- Second choice: Use any linktext -->
    <xsl:when test="*[contains(@class, ' map/topicmeta ')]/*[contains(@class, ' map/linktext ')]"><xsl:value-of select="*[contains(@class, ' map/topicmeta ')]/*[contains(@class, ' map/linktext ')]"/></xsl:when>
    <xsl:otherwise>
      <xsl:choose>
        <!-- Otherwise: fall back to some version of the HREF attribute -->
        <xsl:when test="not(not(@format) or @format='dita' or @format='DITA')"><xsl:value-of select="@href"/></xsl:when>
        <xsl:when test="starts-with(@href,'#')"><xsl:value-of select="@href"/></xsl:when>
        <xsl:when test="contains(@copy-to,'.dita')">
          <xsl:value-of select="substring-before(@copy-to,'.dita')"/>.<xsl:value-of select="$OUTEXT"/>
        </xsl:when>
        <xsl:when test="contains(@href,'.dita')">
          <xsl:value-of select="substring-before(@href,'.dita')"/>.<xsl:value-of select="$OUTEXT"/><xsl:value-of select="substring-after(@href,'.dita')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="@href"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Place a topicref into a list item -->
<xsl:template match="*[contains(@class, ' map/topicref ')]">
  <xsl:value-of select="$newline"/>
  <li>
    <xsl:choose>
      <!-- If there is an href, the title should be a link -->
      <xsl:when test="@href">
        <a>
          <xsl:attribute name="href">
            <xsl:choose>
                 <xsl:when test="(@scope='external' and not(@format)) or not(not(@format) or @format='dita' or @format='DITA')"><xsl:value-of select="@href"/></xsl:when>
                 <xsl:when test="starts-with(@href,'#')"><xsl:value-of select="@href"/></xsl:when>
                 <xsl:when test="contains(@copy-to,'.dita')">
                   <xsl:value-of select="substring-before(@copy-to,'.dita')"/>.<xsl:value-of select="$OUTEXT"/>
                 </xsl:when>
                 <xsl:when test="contains(@href,'.dita')">
                   <xsl:value-of select="substring-before(@href,'.dita')"/>.<xsl:value-of select="$OUTEXT"/><xsl:value-of select="substring-after(@href,'.dita')"/>
                 </xsl:when>
                 <xsl:otherwise>
                   <xsl:value-of select="@href"/>
                 </xsl:otherwise>
            </xsl:choose>
          </xsl:attribute>
          <xsl:call-template name="get-topicref-title"/>
        </a>
      </xsl:when>
      <!-- Otherwise, just use the title text -->
      <xsl:otherwise>
        <xsl:call-template name="get-topicref-title"/>
      </xsl:otherwise>
    </xsl:choose>
    <!-- Put out the HREF value immediately after the title -->
    <xsl:apply-templates select="@href" mode="output-href"/>
    <!-- Process any other attributes -->
    <xsl:apply-templates select="@*"/>
    <!-- Create an ordered or unordered list for children, and then process them -->
    <xsl:choose>
        <xsl:when test="*[not(contains(@class,' map/topicmeta '))] and @collection-type='sequence'">
            <ol><xsl:apply-templates/></ol>
        </xsl:when>
        <xsl:when test="*[not(contains(@class,' map/topicmeta '))]">
            <ul><xsl:apply-templates/></ul>
        </xsl:when>
    </xsl:choose>
  </li>
</xsl:template>

<!-- Put the href attribute into parenthesis -->
<xsl:template match="@href" mode="output-href">
  <xsl:text> (</xsl:text>
  <xsl:value-of select="."/>
  <xsl:text>)</xsl:text>
</xsl:template>

<!-- Put any of these attributes inside brackets. If in a header, do not use
     brackets; instead, put out a break between each attribute. -->
<xsl:template match="@linking|@toc|@print|@format|@type">
  <xsl:if test="not(parent::*[contains(@class, ' map/relcolspec ')])">
    <xsl:text> [</xsl:text>
  </xsl:if>
  <xsl:value-of select="name()"/>
  <xsl:text>="</xsl:text>
  <xsl:value-of select="."/>
  <xsl:text>"</xsl:text>
  <xsl:choose>
    <xsl:when test="parent::*[contains(@class, ' map/relcolspec ')]">
      <br />
    </xsl:when>
    <xsl:otherwise>
      <xsl:text>]</xsl:text>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Put selection attributes inside brackets -->
<xsl:template match="@audience|@platform|@product|@otherprops|@rev">
  <xsl:if test="not(parent::*[contains(@class, ' map/relcolspec ')])">
    <xsl:text> [</xsl:text>
  </xsl:if>
  <xsl:value-of select="name()"/>
  <xsl:text>="</xsl:text>
  <xsl:value-of select="."/>
  <xsl:text>"</xsl:text>
  <xsl:choose>
    <xsl:when test="parent::*[contains(@class, ' map/relcolspec ')]">
      <br />
    </xsl:when>
    <xsl:otherwise>
      <xsl:text>]</xsl:text>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Attributes that we do not handle explicitly are ignored -->
<xsl:template match="@*"/>

</xsl:stylesheet>
