/*
 * Copyright 2006 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.derby.demo.persistence;

import java.io.IOException;
import java.io.InputStream;
import java.util.Date;
import java.util.List;
import java.util.ListIterator;
import java.util.Properties;

import javax.jdo.JDOHelper;
import javax.jdo.PersistenceManager;
import javax.jdo.PersistenceManagerFactory;
import javax.jdo.Query;
import javax.jdo.Transaction;

import org.apache.derby.demo.beans.model.UserBean;
import org.apache.derby.demo.beans.model.UserCreditCardBean;
import org.apache.derby.demo.beans.view.CityBean;
import org.apache.derby.demo.beans.view.FlightHistoryBean;
import org.apache.derby.demo.beans.view.FlightsBean;

public class JPoxJDO {

	private static final String JPOX_CONFIG = "jpox.properties";
	private static Properties prop;
	private boolean isInitialized = false;
	private PersistenceManager pm;

	public JPoxJDO() {
		if (isInitialized) {
			return;
		} else {
			initialize();
		}

	}

	private void initialize() {
		System.out.println("JpoxJDO, initialize called");
		getDatasourceProps();
		pm = createPersistenceManager();
		isInitialized = true;
		return;
	}

	// load all of the various JPOX properties from a properties file

	private void getDatasourceProps() {
		InputStream is = this.getClass().getResourceAsStream(JPOX_CONFIG);

		prop = new Properties();
		try {
			prop.load(is);
		} catch (IOException exception) {

			throw new RuntimeException("jpox.properties not found.",
					exception);

		}
	}

	// create the PersistenceManager using the JDO Helper class
	// the persistencemanage is used for all JPOX transactions

	private static PersistenceManager createPersistenceManager() {

		PersistenceManagerFactory pmf = JDOHelper
				.getPersistenceManagerFactory(prop);

		return pmf.getPersistenceManager();

	}

	// close the persistence manager.

	public void shutdown() {
		if (pm != null) {
			pm.close();
		}

	}

	// If the username is in the APP.USERS table retrieve
	// the database row, which in JPOX is the class UserBean

	public UserBean getUserPassword(String userName) {

		UserBean userBean;

		Transaction txn = pm.currentTransaction();
		try {

			txn.begin();
			// this represents the query:
			// select username, password from APP.USERS where username = ?
			Query query = pm.newQuery(UserBean.class, "userName == username");
			query.declareParameters("String username");
			query.setUnique(true);
			userBean = (UserBean) query.execute(userName.trim());
			pm.retrieve(userBean);
			if (userBean == null) {
				userBean = new UserBean();
			}
			txn.commit();
		} finally {
			if (txn.isActive()) {
				txn.rollback();
			}
		}
		return userBean;

	}

	// Insert a row into the APP.USERS table by first creating a
	// UserBean, then using JPOX to persist the UserBean to Derby.

	public int insertUser(String firstName, String lastName, String userName,
			String email, String password) {

		int success = 0;

		UserBean userBean = new UserBean(firstName.trim(), lastName.trim(),
				userName.trim(), password.trim(), email.trim());

		Transaction txn = pm.currentTransaction();
		try {

			txn.begin();
			// I have to have this, otherwise it is not persistent
			pm.makePersistent(userBean);

			txn.commit();
			success = 1;
		} finally {
			if (txn.isActive()) {
				txn.rollback();
				success = 0;
			}

		}

		return success;
	}

	// insert a row into the UserCreditCard table by first
	// creating a UserCreditCardBean and then using JPOX
	// to persist the data

	public int insertUserCreditCard(String lastname, String userName,
			String creditCardType, String creditCardNum, String creditCardDisplay) {

		int success = 0;

		UserCreditCardBean userCCBean = new UserCreditCardBean(lastname.trim(),
				userName.trim(), creditCardType.trim(), creditCardNum.trim(),
				creditCardDisplay.trim());

		Transaction txn = pm.currentTransaction();
		try {

			txn.begin();
			pm.makePersistent(userCCBean);

			txn.commit();
			success = 1;
		} finally {
			if (txn.isActive()) {
				txn.rollback();
				success = 0;
			}

		}

		return success;
	}

	// query for the Destination Airports of a City based on
	// the Origin Airport of a flight
	// return the data in the form of the JPOX enhanced class,
	// CityBean

	public CityBean[] destAirports(String origAirport) {

		String sql = "select distinct c.city_id, c.city_name, c.country, f.dest_airport "
				+ " from APP.flights f, "
				+ "APP.cities c where "
				+ "f.orig_airport = ? and f.dest_airport = c.airport";

		CityBean[] cityBean;

		Transaction txn = pm.currentTransaction();
		try {

			txn.begin();

			Query query = pm.newQuery("javax.jdo.query.SQL", sql);
			query.setResultClass(CityBean.class);
			List list = (List) query.execute(origAirport.trim());

			ListIterator iter = list.listIterator();

			if (list.size() > 0) {
				cityBean = new CityBean[list.size()];
			} else {
				cityBean = new CityBean[0];
			}
			int i = 0;

			while (iter.hasNext()) {
				cityBean[i++] = (CityBean) iter.next();
			}

			txn.commit();
		} finally {
			if (txn.isActive()) {
				txn.rollback();
			}

		}

		return cityBean;

	}

	// query for all Cities in the APP.CITIES table
	// returning an Array of CityBeans

	public CityBean[] cityList() {

		String sql = "SELECT CITY_ID, CITY_NAME, COUNTRY, AIRPORT FROM APP.CITIES ORDER BY CITY_NAME, COUNTRY";
		CityBean[] cityBean;

		Transaction txn = pm.currentTransaction();
		try {

			txn.begin();

			Query query = pm.newQuery("javax.jdo.query.SQL", sql);
			query.setClass(CityBean.class);
			List list = (List) query.execute();

			ListIterator iter = list.listIterator();

			if (list.size() > 0) {
				cityBean = new CityBean[list.size()];
			} else {
				cityBean = new CityBean[0];
			}
			int i = 0;

			while (iter.hasNext()) {
				cityBean[i++] = (CityBean) iter.next();
			}

			txn.commit();
		} finally {
			if (txn.isActive()) {
				txn.rollback();
			}

		}

		return cityBean;
	}

	// return an array of FlightsBean which include a specific
	// ORIG_AIRPORT and DEST_AIRPORT

	public FlightsBean[] origDestFlightList(String origAirport,
			String destAirport, Date startDate) {

		String sql = "select flight_id, segment_number, orig_airport, depart_time, dest_airport, "
				+ "arrive_time, meal, flying_time, miles,"
				+ " aircraft "
				+ "from APP.FLIGHTS"
				+ " where ORIG_AIRPORT = ? "
				+ " AND DEST_AIRPORT = ? ";

		FlightsBean[] flightsBean;
		Transaction txn = pm.currentTransaction();
		try {

			txn.begin();

			Query query = pm.newQuery("javax.jdo.query.SQL", sql);
			query.setClass(FlightsBean.class);
			List list = (List) query.execute(origAirport.trim(), destAirport.trim());

			ListIterator iter = list.listIterator();

			if (list.size() > 0) {
				flightsBean = new FlightsBean[list.size()];
			} else {
				flightsBean = new FlightsBean[0];
			}
			int i = 0;

			while (iter.hasNext()) {
				flightsBean[i] = (FlightsBean) iter.next();
				flightsBean[i++].setDepartureDate(startDate);

			}

			txn.commit();
		} finally {
			if (txn.isActive()) {
				txn.rollback();
			}

		}

		return flightsBean;
	}

	// return the rows in the FlightHistory table based on the Username
	// in the form of an array of FlightHistoryBeans

	public FlightHistoryBean[] fetchFlightHistory(String userName) {

		FlightHistoryBean[] flightHistoryBeans;

		Transaction txn = pm.currentTransaction();
		try {

			txn.begin();
			Query query = pm
					.newQuery(FlightHistoryBean.class, "userName == username");
			query.declareParameters("String username");
			List list = (List) query.execute(userName);
			pm.retrieveAll(list);

			ListIterator iter = list.listIterator();

			if (list.size() > 0) {
				flightHistoryBeans = new FlightHistoryBean[list.size()];
			} else {
				flightHistoryBeans = new FlightHistoryBean[0];
			}
			int i = 0;

			while (iter.hasNext()) {
				flightHistoryBeans[i++] = (FlightHistoryBean) iter.next();
			}

			txn.commit();
		} finally {
			if (txn.isActive()) {
				txn.rollback();
			}

		}
		return flightHistoryBeans;

	}

	// insert into the UserFlightHistory table by creating
	// the FlightHistoryBean and persisting it to Derby

	public int insertUserFlightHistory(String userName, FlightsBean flightsBean,
			String creditCardType, String creditCardDisplay) {

		int success = 0;

		Transaction txn = pm.currentTransaction();

		try {
			txn.begin();
			FlightHistoryBean flightHistBean = new FlightHistoryBean(userName.trim(),
					flightsBean.getFlightId(), flightsBean.getStartAirport(), flightsBean
							.getDepartureDate().toString().substring(0, 10), flightsBean
							.getEndAirport(), flightsBean.getAircraft(), flightsBean
							.getMiles(), creditCardType.trim(), creditCardDisplay.trim());

			pm.makePersistent(flightHistBean);

			txn.commit();
			success = 1;
		} finally {
			if (txn.isActive()) {
				txn.rollback();
				success = 0;
			}

		}

		return success;

	}

}
